/**
 * @file BF22.h
 * 
 * BF22
 * 
 * This file includes functions for communication with BF22.
 * 
 * BeFREE 22 (BF22) is a touchcomputer with integrated intercom and programmable keys.
 * 
 * Tipro
 * 
 * support@tipro.si
 * 
 * Copyright 2021
 */
 
#ifndef _TIPRO_TIPROHIDAPI_BF22_H_
#define _TIPRO_TIPROHIDAPI_BF22_H_

#include "TiproHidDevice.h"

#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************************/
/*		BF22 DEFINES														*/
/****************************************************************************/

// PCM scenario
#define BF22_PCM_SCENARIO_AB			0x3C /**< PMC scenario - Both channels of codec A to left speaker and both channels of codec B to right speaker. */
#define BF22_PCM_SCENARIO_LR			0x3D /**< PMC scenario - Left channels of codec A and B to left speaker and right channels of codec A and B to right speaker. */
#define BF22_PCM_SCENARIO_VOID			0x3F
#define BF22_PCM_SCENARIO_MASK			0x03

// microphone amplifier mode
#define BF22_MICAMP_MODE_LOG			0x33 /**< Microphone amplifier mode - Logaritmic. When the microphone amplifier is set to logaritmic mode, you can set additional settings: optimal distance, noise threshold and compression. */
#define BF22_MICAMP_MODE_LIN			0x37 /**< Microphone amplifier mode - Linear */
#define BF22_MICAMP_MODE_VOID			0x3F
#define BF22_MICAMP_MODE_MASK			0x0C

// central (low frequency) speaker 
#define BF22_LF_SPK_OFF					0x3C /**< Central speaker disabled. */ 
#define BF22_LF_SPK_ON					0x3D /**< Central speaker activated. */ 
#define BF22_LF_SPK_VOID				0x3F
#define BF22_LF_SPK_MASK				0x03

// showing microphone signal on VU meter
#define BF22_VU_AB_MIC_OFF				0x33 /**< Microphone signal is not shown. */ 
#define BF22_VU_AB_MIC_ON				0x37 /**< Microphone signal is shown on VU meter. */ 
#define BF22_VU_AB_MIC_VOID				0x3F
#define BF22_VU_AB_MIC_MASK				0x0C

// analog line out to speakers
#define BF22_LINE_OUT_OFF_SPK			0x10 /**< Soundcard is not connected to the speakers. */ 
#define BF22_LINE_OUT_ON_L_SPK			0x11 /**< Left channel of soundcard to left speaker. */
#define BF22_LINE_OUT_ON_R_SPK			0x12 /**< Right channel of soundcard to right speaker. */
#define BF22_LINE_OUT_ON_LR_SPK			0x13 /**< Left channel of soundcard to left speaker and right channel to right speaker. */
#define BF22_LINE_OUT_VOID				0x14
#define BF22_LINE_OUT_MASK				0x03

// microphone to analog line in
#define BF22_MIC_OFF_LINE_IN			0x04 /**< Signal from microphone not added to analogue line in. */
#define BF22_MIC_ON_LINE_IN				0x0C /**< Signal from microphone added to analogue line in. */
#define BF22_MIC_LINE_IN_VOID			0x14
#define BF22_MIC_LINE_IN_MASK			0x08

// echo canceller
#define BF22_EC_OFF						0x00 /**< Echo canceller disabled. */
#define BF22_EC_ON_L					0x01 /**< Echo canceller enabled. Left speaker signal as reference. */
#define BF22_EC_ON_R					0x02 /**< Echo canceller enabled. Right speaker signal as reference. */
#define BF22_EC_ON_LR					0x03 /**< Echo canceller enabled. Left and right speaker signal as reference. */
#define BF22_EC_VOID					0x07
#define BF22_EC_MASK					0x03

// PTT LED controll
#define BF22_PTT_LED_OFF				0x00 /**< PTT LED is off. */
#define BF22_PTT_LED_ON					0x10 /**< PTT LED lights. */
#define BF22_PTT_LED_BLINK				0x20 /**< PTT LED blinks. */

// microphone
#define BF22_MIC_ACTIVE					0x00 /**< Microphone is active. */
#define BF22_MIC_MUTE					0x01 /**< Microphone is muted. */

// microphone level (attenuation)
#define BF22_MIC_LEVEL_P3				0x10 /**< Microphone attenuation level +3 (maximum) */
#define BF22_MIC_LEVEL_P2				0x11 /**< Microphone attenuation level +2 */
#define BF22_MIC_LEVEL_P1				0x12 /**< Microphone attenuation level +1 */
#define BF22_MIC_LEVEL_0				0x13 /**< Microphone attenuation level 0 */
#define BF22_MIC_LEVEL_M1				0x14 /**< Microphone attenuation level -1 */
#define BF22_MIC_LEVEL_M2				0x15 /**< Microphone attenuation level -2 */
#define BF22_MIC_LEVEL_M3				0x16 /**< Microphone attenuation level -3 */
#define BF22_MIC_LEVEL_M4				0x17 /**< Microphone attenuation level -4 */
#define BF22_MIC_LEVEL_M5				0x18 /**< Microphone attenuation level -5 */
#define BF22_MIC_LEVEL_M6				0x19 /**< Microphone attenuation level -6 */
#define BF22_MIC_LEVEL_M7				0x1A /**< Microphone attenuation level -7 */
#define BF22_MIC_LEVEL_M8				0x1B /**< Microphone attenuation level -8 */
#define BF22_MIC_LEVEL_M9				0x1C /**< Microphone attenuation level -9 */
#define BF22_MIC_LEVEL_M10				0x1D /**< Microphone attenuation level -10 */
#define BF22_MIC_LEVEL_M11				0x1E /**< Microphone attenuation level -11 (minimum) */

#define BF22_MIC_LEVEL_MIN				BF22_MIC_LEVEL_P3
#define BF22_MIC_LEVEL_MAX				BF22_MIC_LEVEL_M11

//microphone optimal distance
#define BF22_MIC_DISTANCE_M2			0x0A /**< Microphone preamplifier level -2 (minimum) */
#define BF22_MIC_DISTANCE_M1			0x0B /**< Microphone preamplifier level -1 */
#define BF22_MIC_DISTANCE_0				0x0C /**< Microphone preamplifier level 0 */
#define BF22_MIC_DISTANCE_P1			0x0D /**< Microphone preamplifier level +1 */
#define BF22_MIC_DISTANCE_P2			0x0E /**< Microphone preamplifier level +2 */
#define BF22_MIC_DISTANCE_P3			0x0F /**< Microphone preamplifier level +3 */
#define BF22_MIC_DISTANCE_P4			0x10 /**< Microphone preamplifier level +4 */
#define BF22_MIC_DISTANCE_P5			0x11 /**< Microphone preamplifier level +5 */
#define BF22_MIC_DISTANCE_P6			0x12 /**< Microphone preamplifier level +6 */
#define BF22_MIC_DISTANCE_P7			0x13 /**< Microphone preamplifier level +7 */
#define BF22_MIC_DISTANCE_P8			0x14 /**< Microphone preamplifier level +8 (maximum) */

#define BF22_MIC_DISTANCE_VERY_SHORT	BF22_MIC_DISTANCE_0 /**< Very short distance (~10 cm) */
#define BF22_MIC_DISTANCE_SHORT			BF22_MIC_DISTANCE_P2 /**< Short distance (~30 cm) */
#define BF22_MIC_DISTANCE_MEDIUM		BF22_MIC_DISTANCE_P4 /**< Medium distance (~100 cm) */
#define BF22_MIC_DISTANCE_LONG			BF22_MIC_DISTANCE_P6 /**< Long distance (~200 cm) */

#define BF22_MIC_DISTANCE_MIN			BF22_MIC_DISTANCE_M2
#define BF22_MIC_DISTANCE_MAX			BF22_MIC_DISTANCE_P8

// microphone compression
#define BF22_MIC_COMPRESSION_OFF		0x10 /**< Microphone compression off */
#define BF22_MIC_COMPRESSION_ON			0x11 /**< Microphone compression on */
// limits to check
#define BF22_MIC_COMPRESSION_MIN		BF22_MIC_COMPRESSION_OFF
#define BF22_MIC_COMPRESSION_MAX		BF22_MIC_COMPRESSION_ON

// microphone noise threshold
#define BF22_MIC_THRESHOLD_HIGH			0x10 /**< Microphone noise threshold high */
#define BF22_MIC_THRESHOLD_MID			0x11 /**< Microphone noise threshold medium */
#define BF22_MIC_THRESHOLD_LOW			0x12 /**< Microphone noise threshold low */
#define BF22_MIC_THRESHOLD_LOWER		0x13 /**< Microphone noise threshold lower */
#define BF22_MIC_THRESHOLD_VERY_LOW		0x14 /**< Microphone noise threshold very low */
// limits to check
#define BF22_MIC_THRESHOLD_MIN			BF22_MIC_THRESHOLD_HIGH
#define BF22_MIC_THRESHOLD_MAX			BF22_MIC_THRESHOLD_VERY_LOW

//luminance
#define BF22_LUMINANCE_MIN				0x00 /**< LCD screen luminance 0 (minimum) */
#define BF22_LUMINANCE_MAX				0x14 /**< LCD screen luminance 20 (maximum) */

//speaker level
#define BF22_SPK_LEVEL_MIN				0x00 /**< Speaker level 0 (minimum) */
#define BF22_SPK_LEVEL_MAX				0x14 /**< Speaker level 20 (maximum) */

// line-keys LED states
// bit0 - m - set mask
// bit1 - g - green
// bit2 - r - red
// bit3 - G - blink green
// bit4 - R - blink red
//																					//RGrgm
#define BF22_LED_NOCHANGE				0x00 /**< Do not set the LED */				//00000 
#define BF22_LED_OFF					0x01 /**< LED off */						//00001 
#define BF22_LED_GREEN					0x03 /**< LED lights in green colour */		//00011
#define BF22_LED_RED					0x05 /**< LED lights in red colour */		//00101 
#define BF22_LED_ORANGE					0x07 /**< LED lights in orange colour */	//00111
#define BF22_LED_BLINK_OFF_GREEN		0x09 /**< LED blinks (off and green) */		//01001 
#define BF22_LED_BLINK_OFF_RED			0x11 /**< LED blinks (off and red) */		//10001 
#define BF22_LED_BLINK_OFF_ORANGE		0x19 /**< LED blinks (off and orange) */	//11001
#define BF22_LED_BLINK_GREEN_OFF		0x0B /**< LED blinks (green and off) */		//01011 
#define BF22_LED_BLINK_GREEN_RED		0x1B /**< LED blinks (green and red) */		//11011 
#define BF22_LED_BLINK_GREEN_ORANGE		0x13 /**< LED blinks (green and orange) */	//10011 
#define BF22_LED_BLINK_RED_OFF			0x15 /**< LED blinks (red and off) */		//10101 
#define BF22_LED_BLINK_RED_GREEN		0x1D /**< LED blinks (red and green) */		//11101
#define BF22_LED_BLINK_RED_ORANGE		0x0D /**< LED blinks (red and orange) */	//01101  
#define BF22_LED_BLINK_ORANGE_OFF		0x1F /**< LED blinks (orange and off) */	//11111 
#define BF22_LED_BLINK_ORANGE_GREEN		0x17 /**< LED blinks (orange and green) */	//10111
#define BF22_LED_BLINK_ORANGE_RED		0x0F /**< LED blinks (orange and red) */	//01111 

// limits to check
#define BF22_LED_MIN					BF22_LED_NOCHANGE
#define BF22_LED_MAX					BF22_LED_BLINK_ORANGE_OFF


/****************************************************************************/
/*		BF22 FUNCTIONS														*/
/****************************************************************************/

/**
 * @brief Sets PCM scenario
 * @param[in] device Device
 * @param[in] nScenario PCM scenario (AB2LR / LR2LR)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetPCMScenario(TiproHidDevice *device, int nScenario);

/**
 * @brief Gets PCM scenario
 * @param[in] device Device
 * @param[out] nScenario PCM scenario (AB2LR / LR2LR)
 * @param[out] nInitScenario Initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetPCMScenario(TiproHidDevice *device, int *nScenario, int *nInitScenario);

/**
 * @brief Sets Microphone amplifier mode (Linear/Logarithmic)
 * @param[in] device Device
 * @param[in] nMicAmpMode Microphone amplifier mode (LIN/LOG)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicAmpMode(TiproHidDevice *device, int nMicAmpMode);

/**
 * @brief Get Microphone amplifier mode (Linear/Logarithmic)
 * @param[in] device Device
 * @param[out] nMicAmpMode Microphone amplifier mode (LIN/LOG)
 * @param[out] nInitMicAmpMode Initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicAmpMode(TiproHidDevice *device, int *nMicAmpMode, int *nInitMicAmpMode);

/**
 * @brief Set LF speaker state (ON/OFF) 
 * @param[in] device Device
 * @param[in] nState LF speaker state (on/off)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetLFSpeakerState(TiproHidDevice *device, int nState);

/**
 * @brief Get LF speaker state
 * @param[in] device Device
 * @param[out] nState LF speaker state (on/off)
 * @param[out] nInitState Initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetLFSpeakerState(TiproHidDevice *device, int *nState, int *nInitState);

/**
 * @brief Sets if microphone level is shown on the VU-meter (ON/OFF)
 * @param[in] device Device
 * @param[in] nState Microphone on VU-meter state (on/off)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicOnVUState(TiproHidDevice *device, int nState);

/**
 * @brief Get microphone level shown on VU-meter state
 * @param[in] device Device
 * @param[out] nState microphone level on VU-meter state (on/off)
 * @param[out] nInitState Initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicOnVUState(TiproHidDevice *device, int *nState, int *nInitState);

/**
 * @brief Sets the connection state (ON/OFF) of the gooseneck microphone to the LINE-IN port on the motherboard
 * @param[in] device Device
 * @param[in] nState connection state (ON/OFF)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMic2LineInState(TiproHidDevice *device, int nState);

/**
 * @brief Returns the connection state (ON/OFF) of the gooseneck microphone to the LINE-IN port on the motherboard
 * @param[in] device Device
 * @param[out] nState connection state (ON/OFF)
 * @param[out] nInitState initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMic2LineInState(TiproHidDevice *device, int *nState, int *nInitState);

/**
 * @brief Sets the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT port to the speakers.
 * @param[in] device Device
 * @param[in] nMode connection mode (OFF/L/R/LR)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetLineOut2SpkMode(TiproHidDevice *device, int nMode);

/**
 * @brief Returns the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT port to the speakers.
 * @param[in] device Device
 * @param[out] nMode connection mode (OFF/L/R/LR)
 * @param[out] nInitMode initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetLineOut2SpkMode(TiproHidDevice *device, int *nMode, int *nInitMode);

/**
 * @brief Sets the echo canceler mode (OFF/L/R/LR)
 * @param[in] device Device
 * @param[in] nMode echo canceler mode (OFF/L/R/LR)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetEchoMode(TiproHidDevice *device, int nMode);

/**
 * @brief Returns the echo canceler mode (OFF/L/R/LR)
 * @param[in] device Device
 * @param[out] nMode echo canceler mode (OFF/L/R/LR)
 * @param[out] nInitMode initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetEchoMode(TiproHidDevice *device, int *nMode, int *nInitMode);

/**
 * @brief Disables touchscreen
 * @param[in] device Device
 * @return Command status: 0-OK, else error code
 */
int TiproBF22TouchscreenDisable(TiproHidDevice *device);

/**
 * @brief Enables touchscreen
 * @param[in] device Device
 * @return Command status: 0-OK, else error code
 */
int TiproBF22TouchscreenEnable(TiproHidDevice *device);

/**
 * @brief Sets the LED state of illuminated PTT key
 * @param[in] device Device
 * @param[in] nState PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetPTTKeyLEDState(TiproHidDevice *device, int nState);

/**
 * @brief Sets the operation of the illuminated PTT key with more paramters
 * @param[in] device Device
 * @param[in] nState PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)
 * @param[in] nTBlinkOn On period for blink mode (ms)
 * @param[in] nTBlinkOff Off period for blink mode (ms)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetPTTKeyLEDStateEx(TiproHidDevice *device, int nState, int nTBlinkOn, int nTBlinkOff);

/**
 * @brief Returns the LED state of the illuminated PTT Key
 * @param[in] device Device
 * @param[out] nState state of the LED (LED_ON/LED_OFF/LED_BLINK)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetPTTKeyLEDState(TiproHidDevice *device, int *nState);

/**
 * @brief Mutes/un-mutes the microphone signal
 * @param[in] device Device
 * @param[in] nMute mute/un-mute microphone
 * @return Command status: 0-OK, else error code
 */
int TiproBF22MuteMicrophone(TiproHidDevice *device, int nMute);

/**
 * @brief Sets the level of the left speaker
 * @param[in] device Device
 * @param[in] nLevel Speaker level (0 - 20)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetLeftSpeakerLevel(TiproHidDevice *device, int nLevel);

/**
 * @brief Returns the level of the left speaker
 * @param[in] device Device
 * @param[out] nLevel speaker level (0 - 20)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetLeftSpeakerLevel(TiproHidDevice *device, int *nLevel);

/**
 * @brief Sets the level of the right speaker
 * @param[in] device Device
 * @param[in] nLevel speaker level (0 - 20)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetRightSpeakerLevel(TiproHidDevice *device, int nLevel);

/**
 * @brief Returns the level of the right speaker
 * @param[in] device Device
 * @param[out] nLevel speaker level (0 - 20)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetRightSpeakerLevel(TiproHidDevice *device, int *nLevel);

/**
 * @brief Sets the gooseneck microphone level
 * @param[in] device Device
 * @param[in] nLevel microphone level
 * @param[in] bSetInit write initial state to EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicrophoneLevel(TiproHidDevice *device, int nLevel, bool bSetInit);

/**
 * @brief Returns the googeneck microphone level
 * @param[in] device Device
 * @param[out] nLevel microphone level
 * @param[out] nInitLevel initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicrophoneLevel(TiproHidDevice *device, int *nLevel, int *nInitLevel);

/**
 * @brief Sets the microphone optimal distance
 * @param[in] device Device
 * @param[in] nDistance microphone optimal distance
 * @param[in] bSetInit write initial state to EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicrophoneOptimalDistance(TiproHidDevice *device, int nDistance, bool bSetInit);

/**
 * @brief Gets the microphone optimal distance
 * @param[in] device Device
 * @param[out] nDistance microphone optimal distance
 * @param[out] nInitDistance initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicrophoneOptimalDistance(TiproHidDevice *device, int *nDistance, int *nInitDistance);

/**
 * @brief Returns threshold for the microphone signal
 * @param[in] device Device
 * @param[out] nThreshold microphone threshold
 * @param[out] nInitThreshold initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicrophoneThreshold(TiproHidDevice *device, int *nThreshold, int *nInitThreshold);

/**
 * @brief Sets threshold for the microphone signal
 * @param[in] device Device
 * @param[in] nThreshold microphone threshold
 * @param[in] bSetInit write initial state to EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicrophoneThreshold(TiproHidDevice *device, int nThreshold, bool bSetInit);

/**
 * @brief Returns compression for the microphone signal
 * @param[in] device Device
 * @param[out] nCompression microphone compression
 * @param[out] nInitCompression initial state from EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetMicrophoneCompression(TiproHidDevice *device, int *nCompression, int *nInitCompression);

/**
 * @brief Sets compression for the microphone signal
 * @param[in] device Device
 * @param[in] nCompression microphone compression
 * @param[in] bSetInit write initial state to EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetMicrophoneCompression(TiproHidDevice *device, int nCompression, bool bSetInit);

/**
 * @brief Returns luminance for both stripe and LCD backlight
 * @param[in] device Device
 * @param[out] nLuminance luminance index (0-darkest; 20-brightest)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetLuminance(TiproHidDevice *device, int *nLuminance);

/**
 * @brief Sets luminance for both stripe and LCD backlight
 * @param[in] device Device
 * @param[in] nLuminance luminance index (0-darkest; 20-brightest)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetLuminance(TiproHidDevice *device, int nLuminance);

/**
 * @brief Returns the state of the PTT key
 * @param[in] device Device
 * @param[out] nState state of the key (KEY_PRESSED / KEY_RELEASED)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetPTTKeyState(TiproHidDevice *device, int *nState);

/**
 * @brief Sets the LED state for each key LED of Line-keys
 * @param[in] device Device
 * @param[in] nKey1State,nKey2State,nKey3State,nKey4State,nKey5State,nKey6State,nKey7State,nKey8State,nKey9State,nKey10State,nKey11State,nKey12State,nKey13State,nKey14State,nKey15State,nKey16State
 * LED state for each key from left to right (BF22_LED_NOCHANGE/BF22_LED_OFF/BF22_LED_GREEN/BF22_LED_RED/BF22_LED_ORANGE/BF22_LED_BLINK_OFF_GREEN/BF22_LED_BLINK_OFF_RED/BF22_LED_BLINK_OFF_ORANGE/BF22_LED_BLINK_GREEN_OFF/BF22_LED_BLINK_GREEN_RED/BF22_LED_BLINK_GREEN_ORANGE/BF22_LED_BLINK_RED_OFF/BF22_LED_BLINK_RED_GREEN/BF22_LED_BLINK_RED_ORANGE/BF22_LED_BLINK_ORANGE_OFF/BF22_LED_BLINK_ORANGE_GREEN/BF22_LED_BLINK_ORANGE_RED)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22SetLineKeysLedState(TiproHidDevice *device, int nKey1State, int nKey2State, int nKey3State,int nKey4State, int nKey5State, int nKey6State, int nKey7State, int nKey8State, int nKey9State, int nKey10State, int nKey11State, int nKey12State, int nKey13State, int nKey14State, int nKey15State, int nKey16State);

/**
 * @brief Returns the LED state for each key LED of Line-keys
 * @param[in] device Device
 * @param[out] nKey1State,nKey2State,nKey3State,nKey4State,nKey5State,nKey6State,nKey7State,nKey8State,nKey9State,nKey10State,nKey11State,nKey12State,nKey13State,nKey14State,nKey15State,nKey16State
 * LED state of each key from left to right (BF22_LED_OFF/BF22_LED_GREEN/BF22_LED_RED/BF22_LED_ORANGE/BF22_LED_BLINK_OFF_GREEN/BF22_LED_BLINK_OFF_RED/BF22_LED_BLINK_OFF_ORANGE/BF22_LED_BLINK_GREEN_OFF/BF22_LED_BLINK_GREEN_RED/BF22_LED_BLINK_GREEN_ORANGE/BF22_LED_BLINK_RED_OFF/BF22_LED_BLINK_RED_GREEN/BF22_LED_BLINK_RED_ORANGE/BF22_LED_BLINK_ORANGE_OFF/BF22_LED_BLINK_ORANGE_GREEN/BF22_LED_BLINK_ORANGE_RED)
 * @return Command status: 0-OK, else error code
 */
int TiproBF22GetLineKeysLedState(TiproHidDevice *device, int *nKey1State, int *nKey2State, int *nKey3State, int *nKey4State, int *nKey5State, int *nKey6State, int *nKey7State, int *nKey8State, int *nKey9State, int *nKey10State, int *nKey11State, int *nKey12State, int *nKey13State, int *nKey14State, int *nKey15State, int *nKey16State);

#ifdef __cplusplus
}
#endif

#endif
