/**
 * @file BF10.h
 * 
 * BF10
 * 
 * This file includes functions for communication with BF10.
 * 
 * BeFREE 10 (BF10) is a touchcomputer with integrated intercom.
 * 
 * Tipro
 * 
 * support@tipro.si
 * 
 * Copyright 2021
 */
 
#ifndef _TIPRO_TIPROHIDAPI_BF10_H_
#define _TIPRO_TIPROHIDAPI_BF10_H_

#include "TiproHidDevice.h"

#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************************/
/*		BF10 FUNCTIONS														*/
/****************************************************************************/

/**
 * @brief Sets luminance for both stripe and LCD backlight
 * @param[in] device Device 
 * @param[in] nIndex Lumninance index (0-darkest; 10-brightest)
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetLuminance (TiproHidDevice *device, int nIndex);

/**
 * @brief Disables touchscreen
 * @param[in] device Device 
 * @return Command status: 0-OK, else error code
 */
int TiproBF10TouchscreenDisable(TiproHidDevice *device);

/**
 * @brief Enables touchscreen
 * @param[in] device Device
 * @return Command status: 0-OK, else error code
 */
int TiproBF10TouchscreenEnable(TiproHidDevice *device);

/**
 * @brief Sets the level of the speakers
 * @param[in] device Device
 * @param[in] nLevel Speaker level
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetSpeakerLevel(TiproHidDevice *device, int nLevel);

/**
 * @brief Returns the level of the speakers
 * @param[in] device Device 
 * @param[out] nLevel Speaker level
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetSpeakerLevel(TiproHidDevice *device, int *nLevel);

/**
 * @brief Returns the state of the PTT key
 * @param[in] device Device 
 * @param[out] nState State of the key (KEY_PRESSED/KEY_RELEASED)
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetPTTKeyState (TiproHidDevice *device, int *nState);

/**
 * @brief Sets the LED state of illuminated PTT key
 * @param[in] device Device 
 * @param[in] nState State of the LED (LED_ON/LED_OFF/LED_BLINK)
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetPTTKeyLEDState (TiproHidDevice *device, int nState);

/**
 * @brief Sets the operation of the illuminated PTT key with more paramters
 * @param[in] device Device 
 * @param[in] nState State of the LED (LED_ON/LED_OFF/LED_BLINK)
 * @param[in] nTBlinkOn On period for blink mode (ms)
 * @param[in] nTBlinkOff Off period for blink mode (ms)
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetPTTKeyLEDStateEx (TiproHidDevice *device, int nState, int nTBlinkOn, int nTBlinkOff);

/**
 * @brief Returns the LED state of the illuminated PTT Key
 * @param[in] device Device 
 * @param[out] nState State of the LED (LED_ON/LED_OFF/LED_BLINK)
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetPTTKeyLEDState (TiproHidDevice *device, int *nState);

/**
 * @brief Mutes/un-mutes the microphone signal
 * @param[in] device Device 
 * @param[in] nMute Mute/un-mute microphone
 * @return Command status: 0-OK, else error code
 */
int TiproBF10MuteMicrophone (TiproHidDevice *device, int nMute);

/**
 * @brief Sets the level of the microphone signal
 * @param[in] device Device 
 * @param[in] nLevel Microphne level
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetMicrophoneLevel (TiproHidDevice *device, int nLevel);

/**
 * @brief Sets the optimal distance of the handsfree microphone signal
 * @param[in] device Device 
 * @param[in] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetMicrophoneOptimalDistance (TiproHidDevice *device, int nDistance);

/**
 * @brief Sets threshold for the microphone signal
 * @param[in] device Device 
 * @param[in] nThreshold Microphone threshold
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetMicrophoneThreshold (TiproHidDevice *device, int nThreshold);

/**
 * @brief Sets compression for the microphone signal
 * @param[in] device Device 
 * @param[in] nCompression Microphone compression
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetMicrophoneCompression (TiproHidDevice *device, int nCompression);

/**
 * @brief Sets the Analog audio options
 * @param[in] device Device 
 * @param[in] nSettings Select which analog channel is added to the output
 * @return Command status: 0-OK, else error code
 */
int TiproBF10SetAnalogAudio(TiproHidDevice *device, int nSettings);

/**
 * @brief Returns Analog audio options
 * @param[in] device Device 
 * @param[out] nSettings Returns which analog channel is added to the output
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetAnalogAudio(TiproHidDevice *device, int *nSettings);

/**
 * @brief Returns the level of the microphone signal
 * @param[in] device Device 
 * @param[out] nLevel Microphone level
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetMicrophoneLevel (TiproHidDevice *device, int *nLevel);

/**
 * @brief Returns the optimal distance of the handsfree microphone signal
 * @param[in] device Device 
 * @param[out] nRange Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetMicrophoneOptimalDistance (TiproHidDevice *device, int *nRange);

/**
 * @brief Returns threshold for the microphone signal
 * @param[in] device Device 
 * @param[out] nThreshold Microphone threshold
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetMicrophoneThreshold (TiproHidDevice *device, int *nThreshold);

/**
 * @brief Returns compression for the microphone signal
 * @param[in] device Device 
 * @param[out] nCompression Microphone compression
 * @return Command status: 0-OK, else error code
 */
int TiproBF10GetMicrophoneCompression (TiproHidDevice *device, int *nCompression);

/**
 * @brief Enables manual control of microphone
 * @param[in] device Device 
 * @param[in] nEnable Enable / disable manual control
 * @return Command status: 0-OK, else error code
 */
int TiproBF10EnableManualMicrophoneControl(TiproHidDevice *device, int nEnable);

#ifdef __cplusplus
}
#endif

#endif
