/**
 * @file TiproTelephony.h
 * 
 * TiproTelephony
 * 
 * This file includes functions for communication with Tipro HID telephony devices.
 * 
 * You can recive HID telephony events - key presses and releases - by defining a callback function.
 * 
 * Tipro
 * 
 * support@tipro.si
 * 
 * Copyright 2021
 */
 
#ifndef _TIPRO_TIPROHIDAPI_TIPROTELEPHONY_H_
#define _TIPRO_TIPROHIDAPI_TIPROTELEPHONY_H_

#ifdef __cplusplus
extern "C" {
#endif

typedef unsigned short USHORT;
typedef void * PVOID;
typedef unsigned long ULONG;

typedef int (*tfnCallback)(USHORT, USHORT, USHORT, USHORT, bool);
typedef int (*tfnThreadStatus)(USHORT, int, int, int);

/**
 * @brief Scans all USB ports and updates the list of connected telephony devices.
 * This function should be called before using other telephony commands or when new device is connected to the system.
 * @return Command status: 0-OK, else error code
 */

int TiproTelephony_DetectDevices();

/**
 * @brief Returns number of connected telephony devices
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_GetNumOfDetectedDevices();

/**
 * @brief Register callback function for returning pressed/released keys and status of the thread
 * @param hidIx Telephony device index
 * @param fnKeyCallback Key callback function
 * - fnCallback(USHORT nID, USHORT nUsagePage, USHORT nLinkUsage, USHORT nUsage, bool bPressed);
 * - Parameters:
 *   + nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
 *   + nUsagePage;				// top collection usage page (for now always 0x0b = Telephony)
 *   + nLinkUsage;				// usage of the specific link collection (0x01 = Phone, 0x06 = Key Pad, 0x07 = Progrmable Button)
 *   + nUsage;					// usage at the specific collection
 *   + bPressed;				// key pressed = TRUE or released = FALSE
 * @param fnKeyStatusCallback Status callback function
 * - fnKeyStatusCallback(USHORT nID, int nStatus, int nErrCode, int nRFUParam)
 * - Parameters:
 *   + nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
 *   + nStatus;				// Status of the thread ( to be defined later... for instance 0x00 = Stopped, 0x01 = Running ... )
 *   + nErrCode;				// ErrCode for reported status ( to be defined later )
 *   + nRFUParam;				// Reserved for future use ( to be defined later )
 * - fnKeyStatusCallback can be NULL if not needed
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_RegisterCallback(USHORT hidIx, tfnCallback fnKeyCallback, tfnThreadStatus fnKeyStatusCallback);

/**
 * @brief Stops reporting keys of telephony device.
 * @param[in] hidIx Telephony device index
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_Stop(const USHORT nLocalIx);

/**
 * @brief Stops reporting keys of all telephony devices.
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_StopAll();

/**
 * @brief Returns telephony device path that can be used to access the device directly
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (char *) where the path will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_GetDevicePath(USHORT hidIx, PVOID pString, ULONG lBufferLen);

/**
 * @brief Returns telephony device product string
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (wchar_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_GetDeviceProductString(USHORT hidIx, PVOID pString, ULONG lBufferLen);

/**
 * @brief Returns telephony device vendor ID and product ID
 * @param[in] hidIx Telephony device index
 * @param[out] nVID Vendor ID number
 * @param[out] nPID Product ID number
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_GetDeviceVidPid(USHORT hidIx, USHORT *nVID, USHORT *nPID);

/**
 * @brief Returns telephony device manufacturer string
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (wchar_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
int TiproTelephony_GetDeviceManufacturerString(USHORT hidIx, PVOID pString, ULONG lBufferLen);

#ifdef __cplusplus
}
#endif

#endif