// The following ifdef block is the standard way of creating macros which make exporting 
// from a DLL simpler. All files within this DLL are compiled with the TIPROHIDAPI_EXPORTS
// symbol defined on the command line. this symbol should not be defined on any project
// that uses this DLL. This way any other project whose source files include this file see 
// TIPROHIDAPI_API functions as being imported from a DLL, wheras this DLL sees symbols
// defined with this macro as being exported.

#ifndef TIPROHIDAPI_H__
	#define TIPROHIDAPI_H__
	
#ifdef _WIN32
	#ifdef TIPROHIDAPI_EXPORTS
		#define TIPROHIDAPI_API __declspec(dllexport)
	#else
		#define TIPROHIDAPI_API __declspec(dllimport)
	#endif
#else
	#ifdef TIPROHIDAPI_EXPORTS
		#define TIPROHIDAPI_API extern "C"
	#else
		#define TIPROHIDAPI_API extern "C"
	#endif
	
	#define __stdcall /**< __stdcall */
	
	#include <functional>
	
	typedef unsigned short USHORT;
	typedef void * PVOID;
	typedef unsigned long ULONG;
#endif

#define TIPRO_API_VERSION_MAJOR 0 /**< Static/compile-time major version of the library. */
#define TIPRO_API_VERSION_MINOR 0 /**< Static/compile-time minor version of the library. */
#define TIPRO_API_VERSION_BUILD 7 /**< Static/compile-time build version of the library. */

/**
 * @brief Get a runtime version of the library.
 * @param[out] nVersionMajor Major version
 * @param[out] nVersionMinor Minor version
 * @param[out] nVersionBuild Build version
 * @return Error code (0 - no error)
 */
TIPROHIDAPI_API int HIDGetLibraryVersion(int *nVersionMajor, int *nVersionMinor, int *nVersionBuild);


/****************************************************************************/
/*		HIDAPI functions (FW Level 1 required)                              */
/****************************************************************************/
TIPROHIDAPI_API int HIDSetLuminance(int nLum);
/* Sets the luminance of the LCD screen										*/
/* arguments (IN)	: Relative luminance (0-20)								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDTouchscreenDisable(void);
/* Disables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDTouchscreenEnable(void);
/* Enables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */


TIPROHIDAPI_API int HIDSetLeds(int nLEDs);
/* Sets the user-defined LED state											*/
/* arguments (IN)	: LEDs to turn ON/OFF (0x00-0x0F)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDDetectDevices(void);
/* Detect Tipro HID API supported devices									*/
/* return value		: command status : 0 == OK, else error code			    */

/****************************************************************************/
/*		HIDAPI functions (FW Level 2 required)                              */
/****************************************************************************/

TIPROHIDAPI_API int HIDEnumerateModules(void);
/* Enumerates devices on TiproBus											*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDReset(void);
/* Sends Reset to the controller											*/
/* return value		: command status : 0 == OK, else error code			    */

/****************************************************************************/
/*		SPEAKERBOX FUNCTIONS (common for Speakerbox and Intercom)           */
/****************************************************************************/

TIPROHIDAPI_API int HIDSetSpeakerLevel(int nFsNum, int nLevel);
/* Sets the level of the speakers											*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/* compatibility	: Speakerbox and Intercom								*/

TIPROHIDAPI_API int HIDGetSpeakerLevel(int nFsNum, int *nLevel);
/* Returns the level of the speakers										*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/* compatibility	: Speakerbox and Intercom								*/

TIPROHIDAPI_API int HIDGetHandsetState(int nHandsetNum, int *nPTTKey, int *nHookState);
/* Returns the state of the keys											*/
/* arguments (IN)	: selected handset module number	                    */
/* arguments (OUT)	: PTT key state	(KEY_PRESSED / KEY_RELEASED)			*/
/*					: Hook state (KEY_PRESSED / KEY_RELEASED)				*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDGetHandsetStateEx (int nHandsetNum, int *nPTTKey, int *nHookState);
/* Returns the state of the keys											*/
/* arguments (IN)	: selected handset module number	                    */
/* arguments (OUT)	: PTT key state	(KEY_PRESSED / KEY_RELEASED)			*/
/*					: Hook state (KEY_PRESSED / KEY_RELEASED)				*/
/*					: if any of OUT arguments is NULL, handset state is		*/
/*					: sent to system, otherwise just RAW is returned		*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDGetSbxKeyState (int nFsNum, int nKey, int *nState);
/* Returns the state of the keys											*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: key index												*/
/* arguments (OUT)	: state of the key (KEY_PRESSED / KEY_RELEASED)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDSetSbxLEDState (int nFsNum, int nKey, int nState);
/* Sets the LED state of the illuminated keys								*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: illuminated key index									*/
/*					: state of the LED (LED_ON/LED_OFF)						*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/* compatibility	: Speakerbox and Intercom								*/

TIPROHIDAPI_API int HIDGetSbxLEDState (int nFsNum, int nKey, int *nState);
/* Returns the LED state of the illuminated keys							*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: illuminated key index									*/
/* arguments (OUT)	: state of the LED (LED_ON/LED_OFF)						*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/* compatibility	: Speakerbox and Intercom								*/

TIPROHIDAPI_API int HIDSetExternalDeviceLevel (int nFsNum, int nLevel);
/* Sets the level of the connected headset/handset device					*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: level of the signal	 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDGetExternalDeviceLevel (int nFsNum, int *nLevel);
/* Returns the current level of the connected headset/handset device		*/
/* arguments (IN)	: selected speakerbox module number                     */
/* arguments (OUT)	: level of the signal	 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDGetExternalHeadsetState (int nFsNum, int *nState);
/* Returns the state of the headset											*/
/* arguments (IN)	: selected speakerbox module number	                    */
/* arguments (OUT)	: headset state											*/	
/*								(HEADSET_CONNECTED / HEADSET_DISCONNECTED)	*/
/*																			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDGetExternalHeadsetStateEx(int nFsNum, int *nState);
/* Returns the state of the headset											*/
/* arguments (IN)	: selected speakerbox module number	                    */
/* arguments (OUT)	: headset state											*/
/*					    HEADSET_EX_STATE_DISCONNECTED						*/
/*					    HEADSET_EX_STATE_CONNECTED_1						*/
/*					    HEADSET_EX_STATE_CONNECTED_2						*/
/*																			*/
/* return value		: command status : 0 == OK, else error code				*/
 
TIPROHIDAPI_API int HIDMuteMicrophone (int nFsNum, int nMute);
/* Mutes/un-mutes the microphone signal										*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: mute/un-mute microphone 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDSpeakerboxGoOnline (int nFsNum);
/* Puts speakerbox to On-line mode											*/
/* arguments (IN)	: selected speakerbox module number                     */
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/

TIPROHIDAPI_API int HIDIsSpeakerboxOnline (int nFsNum, int *nOnline);
/* Returns On-line / Off-line mode											*/
/* arguments (IN)	: selected speakerbox module number                     */
/* arguments (OUT)	: 0 - Offline ; 1 - Online			                    */
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/

TIPROHIDAPI_API int HIDSetAlertLine (int nFsNum, int nMode);
/* Sets the function (mode) of the alert line								*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: mode (ALERT, VOICE, NOT USED)							*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/*																			*/
/*																			*/
/* ON-Line ONLY!															*/

TIPROHIDAPI_API int HIDGetAlertLine (int nFsNum, int *nMode);
/* returns the function (mode) of the alert line							*/
/* arguments (IN)	: selected speakerbox module number                     */
/* arguments (OUT)	: mode (ALERT, VOICE, NOT USED)							*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/*																			*/
/*																			*/
/* ON-Line ONLY!															*/

TIPROHIDAPI_API int HIDSetActiveDevice (int nFsNum, int nDevice);
/* Sets the device where both speaker and microphone signals				*/
/* will be switched to														*/
/* arguments (IN)	: selected speakerbox module number                     */
/*					: device to be made active								*/
/* return value		: command status : 0 == OK, else error code			    */
/*																			*/
/*																			*/
/*																			*/
/* ON-Line ONLY!															*/

TIPROHIDAPI_API int HIDGetMicrophoneLevel (int nFsNum, int *nLevel);
/* Returns the level of the microphone signal								*/
/* arguments (IN) : selected speakerbox module number						*/
/* arguments (OUT): microphone level										*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneLevel (int nFsNum, int nLevel);
/* Sets the level of the microphone signal									*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone level										*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneLevelEx (int nFsNum, int nLevel, int nDevice);
/* Sets the level of the microphone signal									*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone level										*/
/*				  : device to set (handsfree, handset/headset)				*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDGetMicrophoneOptimalDistance (int nFsNum, int *nDistance);
/* Returns the optimal distance of the handsfree microphone					*/
/* arguments (IN) : selected speakerbox module number						*/
/* arguments (OUT): microphone optimal distance								*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneOptimalDistance (int nFsNum, int nDistance);
/* Sets the optimal distance for the handsfree microphone					*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone optimal distance								*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneThreshold (int nFsNum, int nThreshold);
/* Sets the microphone threshold											*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone threshold									*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDGetMicrophoneThreshold (int nFsNum, int *nThreshold);
/* Returns the microphone threshold											*/
/* arguments (IN) : selected speakerbox module number						*/
/* arguments (OUT): microphone threshold									*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneCompression (int nFsNum, int nCompression);
/* Sets the microphone compression											*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone compression									*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDGetMicrophoneCompression (int nFsNum, int *nCompression);
/* Returns the microphone compression										*/
/* arguments (IN) : selected speakerbox module number						*/
/* arguments (OUT): microphone compression									*/
/* return value   : command status : 0 == OK, else error code				*/

/** Specific device versions **/
TIPROHIDAPI_API int HIDSetMicrophoneThresholdEx (int nFsNum, int nThreshold, int nDevice);
/* Sets the specific device's microphone threshold							*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone threshold									*/
/*				  : device to set (handsfree, handset/headset)				*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDSetMicrophoneCompressionEx (int nFsNum, int nCompression, int nDevice);
/* Sets the specific device's microphone compression					*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : microphone compression									*/
/*				  : device to set (handsfree, handset/headset)				*/
/* return value   : command status : 0 == OK, else error code				*/

/* Supported if firmware with blink functionality							*/
TIPROHIDAPI_API int HIDSetSbxLEDBlinkSpeed (int nFsNum, int nTBlinkA, int nTBlinkB);
/* Sets the LED blink timings												*/
/* arguments (IN) : selected speakerbox module number						*/
/*				  : ON time in ms (1..5000)									*/
/*				  : OFF time in ms (1..5000)								*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDGetSbxLEDBlinkSpeed (int nFsNum, int *nTBlinkA, int *nTBlinkB);
/* Get LED blink timings													*/
/* arguments (IN) : selected speakerbox module number						*/
/* arguments (OUT): ON time in ms (1..5000)									*/
/*				  : OFF time in ms (1..5000)								*/
/* return value   : command status : 0 == OK, else error code				*/

/****************************************************************************/
/*		CHAMELEON FUNCTIONS													*/
/****************************************************************************/

TIPROHIDAPI_API int HIDChmGoOnline (int nChmNum);
/* Puts Chameleon into On-Line operating mode.                              */
/* arguments (IN)  : selected Chameleon module number						*/
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmDisplayIcon (int nChmNum, struct CHM_ICON_DATA IconData);
/* Displays the icon from the library on the specified key.                 */
/* arguments (IN)  : selected Chameleon module number                       */
/*				   : address of the key (0x00-0x0F)                         */
/*				   : video type of the icon (NORMAL or INVERSE)             */
/*				   : index of the icon in the icon library                  */
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmDisplayBitmap (int nChmNum, int nKeyAddr, int nVideoType, struct CHM_BITMAP_DATA BitmapData);
/* Displays the bitmap on the specified key.                 */
/* arguments (IN)  : selected Chameleon module number                       */
/*				   : address of the key (0x00-0x0F)                         */
/*				   : video type of the icon (NORMAL or INVERSE)             */
/*				   : bitmap data							                */
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmChangeBacklighting (int nChmNum, struct CHM_BACKLIGHTING_DATA BacklightingData);
/* Sets the backlighting parameters for the addressed LCD key.              */
/* arguments (IN)  : selected Chameleon module number                       */
/*				   : address of the key (0x00-0x0F)                         */
/*				   : first color for the key backlight			            */
/*				   : alternate color for the key backlight                  */
/*				   : alternating frequency (switching from color1 to color2)*/
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmGoToPage (int nChmNum, int nPageNumber);
/* Loads the selected page settings from the configuration memory.			*/
/* arguments (IN)  : selected Chameleon module number                       */
/*				   : page number					                        */
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmGetPageNumber (int nChmNum, int *nPageNumber);
/* Returns current page number.												*/
/* arguments (IN)  : selected Chameleon module number                       */
/* arguments (OUT) : current page number									*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDChmGetNumberOfPages (int nChmNum, int *nPages);
/* Returns number of preprogrammed pages.									*/
/* arguments (IN)  : selected Chameleon module number                       */
/* arguments (OUT) : number of preprogrammed pages							*/
/* return value    : command status : 0 == OK, else error code			    */

/****************************************************************************/
/*		BF10 FUNCTIONS														*/
/****************************************************************************/
/*																			*/
/* Old functions for BeFree and Speakerbox are updated to support new		*/
/* functionality. Some functions are calling old functions with correct		*/
/* paramters.																*/
/*																			*/
/****************************************************************************/

TIPROHIDAPI_API int HIDBF10SetLuminance (int nIndex);
/* Sets luminance for both stripe and LCD backlight			                */
/* arguments (IN)  : lumninance index (0-darkest; 10-brightest)             */
/* arguments (OUT) : none													*/
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10TouchscreenDisable(void);
/* Disables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10TouchscreenEnable(void);
/* Enables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10SetSpeakerLevel(int nLevel);
/* Sets the level of the speakers											*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10GetSpeakerLevel(int *nLevel);
/* Returns the level of the speakers										*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10GetPTTKeyState (int *nState);
/* Returns the state of the PTT key											*/
/* arguments (OUT)	: state of the key (KEY_PRESSED / KEY_RELEASED)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10SetPTTKeyLEDState (int nState);
/* Sets the LED state of illuminated PTT key								*/
/* arguments (IN)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10SetPTTKeyLEDStateEx (int nState, int nTBlinkOn, int nTBlinkOff);
/* Sets the operation of the illuminated PTT key with more paramters		*/
/* arguments (IN)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/*					: On period for blink mode (ms)							*/
/*					: Off period for blink mode (ms)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10GetPTTKeyLEDState (int *nState);
/* Returns the LED state of the illuminated PTT Key							*/
/* arguments (OUT)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10MuteMicrophone (int nMute);
/* Mutes/un-mutes the microphone signal										*/
/* arguments (IN)	: mute/un-mute microphone 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10SetAutoLuminance (int nAutoMode);
/* Sets luminance for both stripe and LCD backlight to be controlled by		*/
/* ambient light sensor.													*/
/* arguments (IN)	: Auto luminance mode									*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF10GetAutoLuminance (int *nAutoMode);
/* Returns the mode of the auto controlled luminance						*/
/* arguments (OUT)	: Auto luminance mode									*/
/* return value		: command status : 0 == OK, else error code			    */

// functions that are also missing in the speakerbox part
TIPROHIDAPI_API int HIDBF10SetMicrophoneLevel (int nLevel);
/* Sets the level of the microphone signal									*/
/* arguments (IN) : microphne level									        */
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10SetMicrophoneOptimalDistance (int nDistance);
/* Sets the optimal distance of the handsfree microphone signal				*/
/* arguments (IN) : microphone optimal distance								*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10SetMicrophoneThreshold (int nThreshold);
/* Sets threshold for the microphone signal									*/
/* arguments (IN) : microphone threshold									*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10SetMicrophoneCompression (int nCompression);
/* Sets compression for the microphone signal								*/
/* arguments (IN) : microphone compression									*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10SetAnalogAudio(int nSettings);
/* Sets the Analog audio options											*/
/* arguments (IN) : select which analog channel is added to the output		*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10GetAnalogAudio(int *nSettings);
/* Returns Analog audio options												*/
/* arguments (OUT): returns which analog channel is added to the output		*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10GetMicrophoneLevel (int *nLevel);
/* Returns the level of the microphone signal								*/
/* arguments (OUT) : microphone level										*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10GetMicrophoneOptimalDistance (int *nRange);
/* Returns the optimal distance of the handsfree microphone signal			*/
/* arguments (OUT) : microphone optimal distance							*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10GetMicrophoneThreshold (int *nThreshold);
/* Returns threshold for the microphone signal								*/
/* arguments (OUT) : microphone threshold									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10GetMicrophoneCompression (int *nCompression);
/* Returns compression for the microphone signal							*/
/* arguments (OUT) : microphone compression									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF10EnableManualMicrophoneControl(int nEnable);
/* Enables manual control of microphone.									*/
/* If enabled microphone mute and LED are controlled exclusively by API.	*/
/* arguments (IN)	: enable / disable manual control	                    */
/* return value		: command status : 0 == OK, else error code			    */

/****************************************************************************/
/*		BF20 FUNCTIONS														*/
/****************************************************************************/

TIPROHIDAPI_API int HIDBF20SetPCMScenario(int nScenario);
/* Set PCM scenario															*/
/* arguments (IN)	: PCM scenario	(AB2LR / LR2LR)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetPCMScenario(int *nScenario, int *nInitScenario);
/* Get PCM scenario															*/
/* arguments (OUT)	: PCM scenario	(AB2LR / LR2LR)							*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetMicAmpMode(int nMicAmpMode);
/* Set Microphone amplifier mode (Linear/Logarithmic)						*/
/* arguments (IN)	: Microphone amplifier mode (LIN/LOG)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMicAmpMode(int *nMicAmpMode, int *nInitMicAmpMode);
/* arguments (OUT)	: Microphone amplifier mode (LIN/LOG)					*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetLFSpeakerState(int nState);
/* Set LF speaker state (ON/OFF)											*/
/* arguments (IN)	: LF speaker state (on/off)								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetLFSpeakerState(int *nState, int *nInitState);
/* Get LF speaker state														*/
/* arguments (OUT)	: LF speaker state (on/off)								*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetMicOnVUState(int nState);
/* Sets if microphone level is shown on the VU-meter (ON/OFF)				*/
/* arguments (IN)	: microphone on VU-meter state (on/off)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMicOnVUState(int *nState, int *nInitState);
/* Get microphone level shown on VU-meter state								*/
/* arguments (OUT)	: microphone level on VU-meter state (on/off)			*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetMic2LineInState(int nState);
/* Sets the connection state (ON/OFF) of the gooseneck microphone to the	*/
/* LINE-IN port on the motherboard											*/
/* arguments (IN)	: connection state (ON/OFF)								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMic2LineInState(int *nState, int *nInitState);
/* Returns the connection state (ON/OFF) of the gooseneck microphone to the	*/
/* LINE-IN port on the motherboard											*/
/* arguments (OUT)	: connection state (ON/OFF)								*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetLineOut2SpkMode(int nMode);
/* Sets the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT port 	*/
/* to the speakers.															*/
/* arguments (IN)	: connection mode (OFF/L/R/LR)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetLineOut2SpkMode(int *nMode, int *nInitMode);
/* Returns the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT		*/
/* port to the speakers.													*/
/* arguments (OUT)	: connection mode (OFF/L/R/LR)							*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetEchoMode(int nMode);
/* Sets the echo canceler mode (OFF/L/R/LR)									*/
/* arguments (IN)	: echo canceler mode (OFF/L/R/LR)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetEchoMode(int *nMode, int *nInitMode);
/* Returns the echo canceler mode (OFF/L/R/LR)								*/
/* arguments (OUT)	: echo canceler mode (OFF/L/R/LR)						*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20TouchscreenDisable(void);
/* Disables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20TouchscreenEnable(void);
/* Enables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetPTTKeyLEDState(int nState);
/* Sets the LED state of illuminated PTT key								*/
/* arguments (IN)	: PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)	*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetPTTKeyLEDStateEx(int nState, int nTBlinkOn, int nTBlinkOff);
/* Sets the operation of the illuminated PTT key with more paramters		*/
/* arguments (IN)	: PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)	*/
/*					: On period for blink mode (ms)							*/
/*					: Off period for blink mode (ms)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetPTTKeyLEDState(int *nState);
/* Returns the LED state of the illuminated PTT Key							*/
/* arguments (OUT)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20MuteMicrophone(int nMute);
/* Mutes/un-mutes the microphone signal										*/
/* arguments (IN)	: mute/un-mute microphone 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetLeftSpeakerLevel(int nLevel);
/* Sets the level of the left speaker										*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetLeftSpeakerLevel(int *nLevel);
/* Returns the level of the left speaker									*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetRightSpeakerLevel(int nLevel);
/* Sets the level of the right speaker										*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetRightSpeakerLevel(int *nLevel);
/* Returns the level of the right speaker									*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetMicrophoneLevel(int nLevel, bool bSetInit);
/* Sets the gooseneck microphone level										*/
/* arguments (IN)	: microphone level										*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMicrophoneLevel(int *nLevel, int *nInitLevel);
/* Returns the googeneck microphone level									*/
/* arguments (OUT)	: microphone level										*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20SetMicrophoneOptimalDistance(int nDistance, bool bSetInit);
/* Sets the microphone optimal distance										*/
/* arguments (IN)	: microphone optimal distance							*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMicrophoneOptimalDistance(int *nDistance, int *nInitDistance);
/* Gets the microphone optimal distance										*/
/* arguments (OUT)	: microphone optimal distance							*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetMicrophoneThreshold(int *nThreshold, int *nInitThreshold);
/* Returns threshold for the microphone signal								*/
/* arguments (OUT)  : microphone threshold									*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF20SetMicrophoneThreshold(int nThreshold, bool bSetInit);
/* Sets threshold for the microphone signal									*/
/* arguments (IN)   : microphone threshold									*/
/*				    : write initial state to EEPROM							*/
/* return value     : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF20GetMicrophoneCompression(int *nCompression, int *nInitCompression);
/* Returns compression for the microphone signal								*/
/* arguments (OUT)	: microphone compression									*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF20SetMicrophoneCompression(int nCompression, bool bSetInit);
/* Sets compression for the microphone signal									*/
/* arguments (IN)	: microphone compression									*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF20SetLuminance(int nLuminance);
/* Sets luminance for both stripe and LCD backlight			                */
/* arguments (IN)  : luminance index (0-darkest; 20-brightest)             */
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetLuminance(int *nLuminance);
/* Sets luminance for both stripe and LCD backlight			                */
/* arguments (OUT)  : luminance index (0-darkest; 20-brightest)             */
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF20GetPTTKeyState(int *nState);
/* Returns the state of the PTT key											*/
/* arguments (OUT)	: state of the key (KEY_PRESSED / KEY_RELEASED)			*/
/* return value		: command status : 0 == OK, else error code			    */

/****************************************************************************/
/*		BF22 FUNCTIONS														*/
/****************************************************************************/

TIPROHIDAPI_API int HIDBF22SetPCMScenario(int nScenario);
/* Set PCM scenario															*/
/* arguments (IN)	: PCM scenario	(AB2LR / LR2LR)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetPCMScenario(int *nScenario, int *nInitScenario);
/* Get PCM scenario															*/
/* arguments (OUT)	: PCM scenario	(AB2LR / LR2LR)							*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetMicAmpMode(int nMicAmpMode);
/* Set Microphone amplifier mode (Linear/Logarithmic)						*/
/* arguments (IN)	: Microphone amplifier mode (LIN/LOG)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMicAmpMode(int *nMicAmpMode, int *nInitMicAmpMode);
/* arguments (OUT)	: Microphone amplifier mode (LIN/LOG)					*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetLFSpeakerState(int nState);
/* Set LF speaker state (ON/OFF)											*/
/* arguments (IN)	: LF speaker state (on/off)								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetLFSpeakerState(int *nState, int *nInitState);
/* Get LF speaker state														*/
/* arguments (OUT)	: LF speaker state (on/off)								*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetMicOnVUState(int nState);
/* Sets if microphone level is shown on the VU-meter (ON/OFF)				*/
/* arguments (IN)	: microphone on VU-meter state (on/off)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMicOnVUState(int *nState, int *nInitState);
/* Get microphone level shown on VU-meter state								*/
/* arguments (OUT)	: microphone level on VU-meter state (on/off)			*/
/*					: Initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetMic2LineInState(int nState);
/* Sets the connection state (ON/OFF) of the gooseneck microphone to the	*/
/* LINE-IN port on the motherboard											*/
/* arguments (IN)	: connection state (ON/OFF)								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMic2LineInState(int *nState, int *nInitState);
/* Returns the connection state (ON/OFF) of the gooseneck microphone to the	*/
/* LINE-IN port on the motherboard											*/
/* arguments (OUT)	: connection state (ON/OFF)								*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetLineOut2SpkMode(int nMode);
/* Sets the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT port 	*/
/* to the speakers.															*/
/* arguments (IN)	: connection mode (OFF/L/R/LR)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetLineOut2SpkMode(int *nMode, int *nInitMode);
/* Returns the connection mode (OFF/L/R/LR) of the motherboard LINE-OUT		*/
/* port to the speakers.													*/
/* arguments (OUT)	: connection mode (OFF/L/R/LR)							*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetEchoMode(int nMode);
/* Sets the echo canceler mode (OFF/L/R/LR)									*/
/* arguments (IN)	: echo canceler mode (OFF/L/R/LR)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetEchoMode(int *nMode, int *nInitMode);
/* Returns the echo canceler mode (OFF/L/R/LR)								*/
/* arguments (OUT)	: echo canceler mode (OFF/L/R/LR)						*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22TouchscreenDisable(void);
/* Disables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22TouchscreenEnable(void);
/* Enables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetPTTKeyLEDState(int nState);
/* Sets the LED state of illuminated PTT key								*/
/* arguments (IN)	: PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)	*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetPTTKeyLEDStateEx(int nState, int nTBlinkOn, int nTBlinkOff);
/* Sets the operation of the illuminated PTT key with more paramters		*/
/* arguments (IN)	: PTT LED state (PTT_LED_ON/PTT_LED_OFF/PTT_LED_BLINK)	*/
/*					: On period for blink mode (ms)							*/
/*					: Off period for blink mode (ms)						*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetPTTKeyLEDState(int *nState);
/* Returns the LED state of the illuminated PTT Key							*/
/* arguments (OUT)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22MuteMicrophone(int nMute);
/* Mutes/un-mutes the microphone signal										*/
/* arguments (IN)	: mute/un-mute microphone 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetLeftSpeakerLevel(int nLevel);
/* Sets the level of the left speaker										*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetLeftSpeakerLevel(int *nLevel);
/* Returns the level of the left speaker									*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetRightSpeakerLevel(int nLevel);
/* Sets the level of the right speaker										*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetRightSpeakerLevel(int *nLevel);
/* Returns the level of the right speaker									*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetMicrophoneLevel(int nLevel, bool bSetInit);
/* Sets the gooseneck microphone level										*/
/* arguments (IN)	: microphone level										*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMicrophoneLevel(int *nLevel, int *nInitLevel);
/* Returns the googeneck microphone level									*/
/* arguments (OUT)	: microphone level										*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetMicrophoneOptimalDistance(int nDistance, bool bSetInit);
/* Sets the microphone optimal distance										*/
/* arguments (IN)	: microphone optimal distance							*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMicrophoneOptimalDistance(int *nDistance, int *nInitDistance);
/* Gets the microphone optimal distance										*/
/* arguments (OUT)	: microphone optimal distance							*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetMicrophoneThreshold(int *nThreshold, int *nInitThreshold);
/* Returns threshold for the microphone signal								*/
/* arguments (OUT)  : microphone threshold									*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF22SetMicrophoneThreshold(int nThreshold, bool bSetInit);
/* Sets threshold for the microphone signal									*/
/* arguments (IN)   : microphone threshold									*/
/*				    : write initial state to EEPROM							*/
/* return value     : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF22GetMicrophoneCompression(int *nCompression, int *nInitCompression);
/* Returns compression for the microphone signal								*/
/* arguments (OUT)	: microphone compression									*/
/*					: initial state from EEPROM								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF22SetMicrophoneCompression(int nCompression, bool bSetInit);
/* Sets compression for the microphone signal									*/
/* arguments (IN)	: microphone compression									*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF22SetLuminance(int nLuminance);
/* Sets luminance for both stripe and LCD backlight			                */
/* arguments (IN)  : luminance index (0-darkest; 20-brightest)             */
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetLuminance(int *nLuminance);
/* Sets luminance for both stripe and LCD backlight			                */
/* arguments (OUT)  : luminance index (0-darkest; 20-brightest)             */
/* return value    : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22GetPTTKeyState(int *nState);
/* Returns the state of the PTT key											*/
/* arguments (OUT)	: state of the key (KEY_PRESSED / KEY_RELEASED)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF22SetLineKeysLedState(int nKey1State, int nKey2State, 
	int nKey3State,	int nKey4State, int nKey5State, int nKey6State, int nKey7State, 
	int nKey8State, int nKey9State, int nKey10State, int nKey11State, int nKey12State, 
	int nKey13State, int nKey14State, int nKey15State, int nKey16State);
/* Sets the LED state for each key LED of Line-keys									*/
/* arguments (IN)	: LED state (for each key - 16 keys)							*/
/*	(BF22_LED_NOCHANGE/BF22_LED_OFF/BF22_LED_GREEN/BF22_LED_RED/BF22_LED_ORANGE/	*/
/*	BF22_LED_BLINK_OFF_GREEN/BF22_LED_BLINK_OFF_RED/BF22_LED_BLINK_OFF_ORANGE/		*/
/*	BF22_LED_BLINK_GREEN_OFF/BF22_LED_BLINK_GREEN_RED/BF22_LED_BLINK_GREEN_ORANGE/	*/
/*	BF22_LED_BLINK_RED_OFF/BF22_LED_BLINK_RED_GREEN/BF22_LED_BLINK_RED_ORANGE/		*/
/*	BF22_LED_BLINK_ORANGE_OFF/BF22_LED_BLINK_ORANGE_GREEN/BF22_LED_BLINK_ORANGE_RED)*/
/* return value		: command status : 0 == OK, else error code						*/

TIPROHIDAPI_API int HIDBF22GetLineKeysLedState(int *nKey1State, int *nKey2State, 
	int *nKey3State, int *nKey4State, int *nKey5State, int *nKey6State, int *nKey7State, 
	int *nKey8State, int *nKey9State, int *nKey10State, int *nKey11State, int *nKey12State, 
	int *nKey13State, int *nKey14State, int *nKey15State, int *nKey16State);
/* Returns the LED state for each key LED of Line-keys								*/
/* arguments (OUT)	: LED state (for each key - 16 keys)							*/
/*	(BF22_LED_OFF/BF22_LED_GREEN/BF22_LED_RED/BF22_LED_ORANGE/						*/
/*	BF22_LED_BLINK_OFF_GREEN/BF22_LED_BLINK_OFF_RED/BF22_LED_BLINK_OFF_ORANGE/		*/
/*	BF22_LED_BLINK_GREEN_OFF/BF22_LED_BLINK_GREEN_RED/BF22_LED_BLINK_GREEN_ORANGE/	*/
/*	BF22_LED_BLINK_RED_OFF/BF22_LED_BLINK_RED_GREEN/BF22_LED_BLINK_RED_ORANGE/		*/
/*	BF22_LED_BLINK_ORANGE_OFF/BF22_LED_BLINK_ORANGE_GREEN/BF22_LED_BLINK_ORANGE_RED)*/
/* return value		: command status : 0 == OK, else error code						*/


/****************************************************************************/
/*		Handset functions													*/
/****************************************************************************/

TIPROHIDAPI_API int HIDHandsetSetMicrophoneCompression (int nHandsetNum, int nCompression);
/* Set compression for the microphone signal								*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone compression									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetMicrophoneThreshold (int nHandsetNum, int nThreshold);
/* Set compression for the microphone signal								*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone threshold									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetMicrophoneLevel (int nHandsetNum, int nLevel);
/* Set compression for the microphone signal								*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone level										*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetMicrophoneLevelEx(int nHandsetNum, int nLevel, bool bSetInit);
/* Set level of the microphone signal, optionaly store it to EEPROM			*/
/* arguments (IN)	: selected handset module number						*/
/*					: microphone level										*/
/*					: write initial state to EEPROM							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetGetMicrophoneCompression (int nHandsetNum, int *nCompression);
/* Returns compression for the microphone signal							*/
/* arguments (IN) : selected handset module number							*/
/* arguments (OUT) : microphone compression									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetGetMicrophoneThreshold (int nHandsetNum, int *nThreshold);
/* Returns compression for the microphone signal							*/
/* arguments (IN) : selected handset module number							*/
/* arguments (OUT) : microphone threshold									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetGetMicrophoneLevel (int nHandsetNum, int *nLevel);
/* Returns compression for the microphone signal							*/
/* arguments (IN) : selected handset module number							*/
/* arguments (OUT) : microphone level										*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetSpeakerLevel(int nHandsetNum, int nLevel);
/* Sets the level of the handset speaker									*/
/* arguments (IN) : selected handset module number							*/
/*				  : speaker level			 								*/
/* return value	  : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetGetSpeakerLevel(int nHandsetNum, int *nLevel);
/* Returns the level of the handset speaker									*/
/* arguments (IN) : selected handset module number							*/
/* arguments (OUT)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetStoreMicrophoneCompression(int nHandsetNum, int nCompression);
/* Sets and stores compression for the microphone signal in EEPROM			*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone compression									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetStoreMicrophoneThreshold(int nHandsetNum, int nThreshold);
/* Sets and stores thresholed for the microphone signal						*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone threshold									*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetStoreMicrophoneLevel(int nHandsetNum, int nLevel);
/* Sets and stores microphone signal level									*/
/* arguments (IN) : selected handset module number							*/
/*				  : microphone level										*/
/* return value  : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetStoreSpeakerLevel(int nHandsetNum, int nLevel);
/* Sets and stores the level of the handset speaker							*/
/* arguments (IN) : selected handset module number							*/
/*				  : speaker level			 								*/
/* return value	  : command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetSendHookPttEvents(int nHandsetNum, int nTelephonyEvents, int nTelephonyKeys, int nTelephonyButtons);
/* Resets the HID telephony events and sends hook and ptt button events		*/
/* according to their current state											*/
/* arguments (IN) : selected handset module number							*/
/*				  : telephony events to be reset (1=hook, 2=mute)			*/
/*				  : should telephony keys be reset (0=no, else=yes)			*/
/*				  : should telephony program buttons be reset (0=no, else=yes)	*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetMicrophoneOptimalDistance(int nHandsetNum, int nDistance);
/* Sets the microphone optimal distance                                           */
/* arguments (IN)    : address of the module                                       */
/*                                : microphone optimal distance                   */
/* return value            : command status : 0 == OK, else error code            */
/*                                                                                */

TIPROHIDAPI_API int HIDHandsetSetMicAmpMode(int nHandsetNum, int nMicAmpMode);
/* Set Microphone amplifier mode (Linear/Logarithmic)                             */
/* arguments (IN)    : address of the module                                      */
/*                                : Microphone amplifier mode (LIN/LOG)           */
/* return value            : command status : 0 == OK, else error code            */
/*                                                                                 */

TIPROHIDAPI_API int HIDHandsetGetMicrophoneOptimalDistance(int nHandsetNum, int *nDistance);
/* Returns the optimal distance of the microphone                                 */
/* arguments (IN)    : selected handset module number                              */
/* arguments (OUT)   : microphone optimal distance                                 */
/* return value            : command status : 0 == OK, else error code            */
/*                                                                                */

TIPROHIDAPI_API int HIDHandsetStoreMicrophoneOptimalDistance(int nHandsetNum, int nDistance);
/* Sets and stores the optimal distance for the microphone                        */
/* arguments (IN)    : selected handset module number                              */
/*                                : microphone optimal distance                   */
/* return value            : command status : 0 == OK, else error code            */
/*                                                                                */


/****************************************************************************/
/*		Handset functions ( HAA specific )									*/
/****************************************************************************/

TIPROHIDAPI_API int HIDHandsetHAAEnableOnHookSettings(int nHandsetNum);
/* Enables setting microphone and speaker level when handset is on-hook		*/
/* arguments (IN)	: selected handset module number						*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAASetMicrophoneOptimalDistance(int nHandsetNum, int nDistance);
/* Sets the optimal distance for the microphone								*/
/* arguments (IN)	: selected handset module number						*/
/*					: microphone optimal distance							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAAGetMicrophoneOptimalDistance(int nHandsetNum, int *nDistance);
/* Returns the optimal distance of the microphone							*/
/* arguments (IN)	: selected handset module number						*/
/* arguments (OUT)	: microphone optimal distance							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAASetSidetoneLevel(int nHandsetNum, int nLevel);
/* Sets the level of the handset sidetone				 					*/
/* arguments (IN)	: selected handset module number						*/
/*					: handset sidetone level 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetHAASetHandsetSidetone(int nHandsetNum, bool bOn);
/* Turn sidetone On or Off				 									*/
/* arguments (IN)	: selected handset module number						*/
/*					: handset On/Off		 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetHAAGetSidetoneLevel(int nHandsetNum, int *nLevel);
/* Returns the current level of the handset sidetone						*/
/* arguments (IN)	: selected handset module number						*/
/* arguments (OUT)	: handset sidetone level 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetHAAStoreMicrophoneOptimalDistance(int nHandsetNum, int nDistance);
/* Sets and stores the optimal distance for the microphone					*/
/* arguments (IN)	: selected handset module number						*/
/*					: microphone optimal distance							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAAStoreSidetoneLevel(int nHandsetNum, int nLevel);
/* Sets and stores the level of the handset sidetone				 		*/
/* arguments (IN)	: selected handset module number						*/
/*					: handset sidetone level 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDHandsetHAASetForceHFREE(int nHandsetNum);
/* Forces HAA state to HFREE regardless of the hook state					*/
/* arguments (IN)	: selected handset module number						*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAAClearForceHFREE(int nHandsetNum);
/* Clears forced HAA state													*/
/* arguments (IN)	: selected handset module number						*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAAEmulateOnHook(int nHandsetNum);
/* Emulates On-Hook action. The handset has to be programmed properly		*/
/* with ChangeMe for this function to work correctly.						*/
/* arguments (IN)	: selected handset module number						*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetHAAEmulateOffHook(int nHandsetNum);
/* Emulates Off-Hook action. The handset has to be programmed properly		*/
/* with ChangeMe for this function to work correctly.						*/
/* arguments (IN)	: selected handset module number						*/
/* return value		: command status : 0 == OK, else error code				*/


/****************************************************************************/
/*		Handset functions ( Audio monitoring version specific )				*/
/****************************************************************************/

TIPROHIDAPI_API int HIDHandsetSetAMLevel(int nHandsetNum, int nLevel);
/* Set audio monitoring amplification level									*/
/* arguments (IN) : selected handset module number							*/
/*				  : audio monioring amplification level						*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetGetAMLevel(int nHandsetNum, int *nLevel);
/* Returns audio monitoring amplification level								*/
/* arguments (IN)  : selected handset module number							*/
/* arguments (OUT) : audio monioring amplification level					*/
/* return value    : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetSetMicMixLevel(int nHandsetNum, int nLevel);
/* Set near-end microphone signal attenuation level							*/
/* arguments (IN) : selected handset module number							*/
/*				  : near-end microphone signal attenuation level			*/
/* return value   : command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDHandsetGetMicMixLevel(int nHandsetNum, int *nLevel);
/* Returns near-end microphone signal attenuation level						*/
/* arguments (IN)  : selected handset module number							*/
/* arguments (OUT) : near-end microphone signal attenuation level			*/
/* return value    : command status : 0 == OK, else error code				*/


/****************************************************************************/
/*		BF07 FUNCTIONS														*/
/****************************************************************************/

TIPROHIDAPI_API int HIDBF07SetSpeakerLevel(int nLevel);
/* Sets the level of the speakers											*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetSpeakerLevel(int *nLevel);
/* Returns the level of the speakers										*/
/* arguments (IN)	: speaker level			 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetPTTLEDState(int nState);
/* Sets the LED state of PTT key											*/
/*					: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetPTTLEDState(int *nState);
/* Returns the LED state of the PTT key										*/
/* arguments (OUT)	: state of the LED (LED_ON/LED_OFF/LED_BLINK)			*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetHandsetSpeakerLevel(int nLevel);
/* Sets the level of the handset spekaer.				 					*/
/* arguments (IN)	: handset speaker level	 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetHandsetSpeakerLevel(int *nLevel);
/* Returns the current level of the handset speaker							*/
/* arguments (OUT)	: handset speaker level	 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetAlertLine(int nMode);
/* Sets the function (mode) of the alert line								*/
/* arguments (IN)	: mode (ALERT, VOICE, NOT USED)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetAlertLine(int *nMode);
/* returns the function (mode) of the alert line							*/
/* arguments (OUT)	: mode (ALERT, VOICE, NOT USED)							*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetMicrophoneLevel(int nLevel, int nDevice);
/* Sets the level of the microphone signal									*/
/* arguments (IN)	: microphone level										*/
/*					: device to set (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07GetMicrophoneLevel(int *nLevel, int nDevice);
/* Returns the level of microphone signal									*/
/* arguments (OUT)	: microphone level										*/
/* arguments (IN)   : device to get (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07SetMicrophoneThreshold(int nThreshold, int nDevice);
/* Sets the microphone threshold											*/
/* arguments (IN)	: microphone threshold									*/
/*					: device to set (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07GetMicrophoneThreshold(int *nThreshold, int nDevice);
/* Returns microphone threshold											*/
/* arguments (IN)	: microphone threshold									*/
/*					: device to get (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07SetMicrophoneCompression(int nCompression, int nDevice);
/* Sets the microphone compression											*/
/* arguments (IN)	: microphone compression								*/
/*					: device to set (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07GetMicrophoneCompression(int *nCompression, int nDevice);
/* Returns microphone compression											*/
/* arguments (IN)	: microphone compression								*/
/*					: device to set (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07SetMicrophoneOptimalDistance(int nDistance, int nDevice);
/* Sets the optimal distance for the microphone								*/
/* arguments (IN)	: microphone optimal distance							*/
/*					: device to set (handsfree, handset)					*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07GetMicrophoneOptimalDistance(int *nDistance, int nDevice);
/* Returns the optimal distance of the microphone							*/
/* arguments (IN)	: device to set (handsfree, handset)					*/
/* arguments (OUT)	: microphone optimal distance							*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07SetLEDBlinkSpeed(int nTBlinkA, int nTBlinkB);
/* Sets the LED blink timings												*/
/* arguments (IN)   : ON time in ms (1..65000)								*/
/*					: OFF time in ms (1..65000)								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07GetLEDBlinkSpeed(int *nTBlinkA, int *nTBlinkB);
/* Get LED blink timings													*/
/* arguments (IN)	: selected speakerbox module number						*/
/* arguments (OUT)	: ON time in ms (1..65000)								*/
/*					: OFF time in ms (1..65000)								*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07TouchscreenDisable(void);
/* Disables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07TouchscreenEnable(void);
/* Enables touchscreen														*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07DisableEcho(void);
/* Disables echo canceler													*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07EnableEcho(void);
/* Enables echo canceler if present											*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetEchoState(int *nState);
/* Returns echo state														*/
/* arguments (OUT)	: echo state (ON/OFF/NOT_PRESENT)						*/
/* return value		: command status : 0 == OK, else error code				*/

TIPROHIDAPI_API int HIDBF07SetMicOnVUState(int nState);
/* Sets if microphone level is shown on the VU-meter (ON/OFF)				*/
/* arguments (IN)	: microphone on VU-meter state (ON/OFF)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetMicOnVUState(int *nState);
/* Returns if microphone level is shown on the VU-meter (ON/OFF)			*/
/* arguments (OUT)	: microphone on VU-meter state (ON/OFF)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetHandsetSidetoneLevel(int nLevel);
/* Sets the level of the handset sidetone				 					*/
/* arguments (IN)	: handset sidetone level 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SetHandsetSidetone(bool bOn);
/* Turn sidetone On or Off				 									*/
/* arguments (IN)	: handset On/Off		 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetHandsetSidetoneLevel(int *nLevel);
/* Returns the current level of the handset sidetone						*/
/* arguments (OUT)	: handset sidetone level 								*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07EmulateKeyPress(int nKey);
/* Emulates handset or handsfree PTT key press								*/
/* arguments (IN)	: which key to "press" (HS_PTT, HF_PTT)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07EmulateKeyRelease(int nKey);
/* Emulates handset or handsfree PTT key release							*/
/* arguments (IN)	: which key to "press" (HS_PTT, HF_PTT)					*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07GetKeyState(int *nHandsetKeyState, int *nHookState, int *nKey1State, int *nKey2State, int *nKey3State, int *nKey4State, int *nKey5State, int *nKey6State, int *nKeyPttState);
/* Returns the state of the keys											*/
/* arguments (OUT)	: Handset PTT key state	(BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Hook state (BF07_HANDSET_ONHOOK / BF07_HANDSET_OFFHOOK)	*/
/*					: Key 1 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Key 2 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Key 3 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Key 4 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Key 5 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: Key 6 state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/*					: PTT key state (BF07_KEY_PRESSED / BF07_KEY_RELEASED)	*/
/* return value		: command status : 0 == OK, else error code			    */

TIPROHIDAPI_API int HIDBF07SendHookPttEvents(int nTelephonyEvents, int nTelephonyKeys, int nTelephonyButtons);
/* Resets the HID telephony events and sends hook and ptt button events		*/
/* according to their current state											*/
/* arguments (IN) : telephony events to be reset (1=hook, 2=mute)			*/
/*				  : should telephony keys be reset (0=no, else=yes)			*/
/*				  : should telephony program buttons be reset (0=no, else=yes)	*/
/* return value   : command status : 0 == OK, else error code				*/


/****************************************************************************/
/*		SpeakerBox 104 functions											*/
/****************************************************************************/
// Audio output level functions
TIPROHIDAPI_API int HIDSBX104SetAudioOutputLevel(int nAudioOutput, int nLevel);
TIPROHIDAPI_API int HIDSBX104GetAudioOutputLevel(int nAudioOutput, int *nLevel);

// LED controll functions
TIPROHIDAPI_API int HIDSBX104SetLEDState(int nLed, int nState);
TIPROHIDAPI_API int HIDSBX104GetLEDState(int nLed, int *nState);

// Microphone controll functions
TIPROHIDAPI_API int HIDSBX104SetMicrophoneThreshold(int nThreshold);
TIPROHIDAPI_API int HIDSBX104GetMicrophoneThreshold(int *nThreshold);
TIPROHIDAPI_API int HIDSBX104SetMicrophoneCompression(int nCompression);
TIPROHIDAPI_API int HIDSBX104GetMicrophoneCompression(int *nCompression);
TIPROHIDAPI_API int HIDSBX104SetMicrophonePreAmp(int nPreAmp);
TIPROHIDAPI_API int HIDSBX104GetMicrophonePreAmp(int *nPreAmp);
TIPROHIDAPI_API int HIDSBX104SetMicrophoneLevel(int nLevel);
TIPROHIDAPI_API int HIDSBX104GetMicrophoneLevel(int *nLevel);



/****************************************************************************/
/*		HIDAPI telephony functions (HID telephony hardware required)        */
/****************************************************************************/

/**
 * @brief Scans all USB ports and updates the list of connected telephony devices. This function should be called before using other telephony commands or when new device is connected to the system.
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDDetectTelephonyDevices();

/**
 * @brief Returns number of connected telephony devices
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetNumOfDetectedTelephonyDevices();

/**
 * @brief Returns telephony device product string
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetTelephonyDeviceProductString(USHORT hidIx, PVOID pString, ULONG lBufferLen);

/**
 * @brief Returns telephony device vendor ID and product ID
 * @param[in] hidIx Telephony device index
 * @param[out] nVID Vendor ID number
 * @param[out] nPID Product ID number
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetTelephonyDevice_VID_PID(USHORT hidIx, USHORT *nVID, USHORT *nPID);

/**
 * @brief Returns telephony device manufacturer string
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetTelephonyDeviceManufacturerString(USHORT hidIx, PVOID pString, ULONG lBufferLen);

// currently not supported in hardware, not to be used
TIPROHIDAPI_API int HIDGetTelephonyDeviceSNString(USHORT hidIx, PVOID pString, ULONG lBufferLen);


#if 1	// new combined function for registring callbacks
// callback functions for returning pressed/released keys and status of the thread: 
//
// fnCallback(USHORT nID, USHORT nUsagePage, USHORT nLinkUsage, USHORT nUsage, bool bPressed);
//
// Parameters:
//		nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
//		nUsagePage;				// top collection usage page (for now always 0x0b = Telephony)
//		nLinkUsage;				// usage of the specific link collection (0x01 = Phone, 0x06 = Key Pad, 0x07 = Progrmable Button)
//		nUsage;					// usage at the specific collection
//		bPressed;				// key pressed = TRUE or released = FALSE
//
// fnKeyStatusCallback(USHORT nID, int nStatus, int nErrCode, int nRFUParam);
//
// Parameters:
//		nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
//		nStatus;				// Status of the thread ( to be defined later... for instance 0x00 = Stopped, 0x01 = Running ... )
//		nErrCode;				// ErrCode for reported status ( to be defined later )
//		nRFUParam;				// Reserved for future use ( to be defined later )
//
// fnKeyStatusCallback can be NULL if not needed
//
// If Visual Studio 2010 or later is used callback can be defined inside of the class using std:bind

	typedef std::function<int(USHORT,USHORT,USHORT,USHORT,bool)> tfnCallbackEx;
	typedef std::function<int(USHORT,int,int,int)> tfnThreadStatusEx;
	TIPROHIDAPI_API int HIDRegisterTelephonyCallbacksEx(USHORT hidIx, tfnCallbackEx fnKeyCallback, tfnThreadStatusEx fnKeyStatusCallback);

#if _MSC_VER > 1500
	#define _VARIADIC_MAX 10
	#include <functional>
#if 1 // bool in callback - orignal version
	typedef std::function<int(USHORT,USHORT,USHORT,USHORT,bool)> tfnCallbackEx;
#else // bool in callback - USHORT version
	typedef std::function<int(USHORT, USHORT, USHORT, USHORT, USHORT)> tfnCallbackEx;
#endif
	typedef std::function<int(USHORT,int,int,int)> tfnThreadStatusEx;
	TIPROHIDAPI_API int HIDRegisterTelephonyCallbacksEx(USHORT hidIx, tfnCallbackEx fnKeyCallback, tfnThreadStatusEx fnKeyStatusCallback);
#endif
//
// example how to register callback function that is part of a class:
//
//	using namespace std::placeholders;
//  HIDRegisterTelephonyKeysCallbackEx(0,std::bind(&CExampleDlg::OnTelKeyPress,this,_1,_2,_3,_4,_5),NULL);
//
// else normal callback function defintion
#if 1 // bool in callback - orignal version
typedef int (__stdcall *tfnCallback)(USHORT, USHORT, USHORT, USHORT, bool);
#else // bool in callback - USHORT version
typedef int(__stdcall *tfnCallback)(USHORT, USHORT, USHORT, USHORT, USHORT);
#endif
typedef int (__stdcall *tfnThreadStatus)(USHORT, int, int, int);
TIPROHIDAPI_API int HIDRegisterTelephonyCallbacks(USHORT hidIx, tfnCallback fnKeyCallback, tfnThreadStatus fnKeyStatusCallback);

#else
// callback function for returning pressed/released keys: fnCallback(USHORT nID, USHORT nUsagePage, USHORT nLinkUsage, USHORT nUsage, bool bPressed);
//
// Parameters:
//		nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
//		nUsagePage;				// top collection usage page (for now always 0x0b = Telephony)
//		nLinkUsage;				// usage of the specific link collection (0x01 = Phone, 0x06 = Key Pad, 0x07 = Progrmable Button)
//		nUsage;					// usage at the specific collection
//		bPressed;				// key pressed = TRUE or released = FALSE
//
// If Visual Studio 2010 or later is used callback can be defined inside of the class using std:bind
#if _MSC_VER > 1500
	#define _VARIADIC_MAX 10
	#include <functional>
	typedef std::function<int(USHORT,USHORT,USHORT,USHORT,bool)> tfnCallbackEx;
	TIPROHIDAPI_API int HIDRegisterTelephonyKeysCallbackEx(USHORT hidIx, tfnCallbackEx fnKeyCallback);
#endif
//
// example how to register callback function that is part of a class:
//
//	using namespace std::placeholders;
//  HIDRegisterTelephonyKeysCallbackEx(0,std::bind(&CExampleDlg::OnTelKeyPress,this,_1,_2,_3,_4,_5));
//
// else normal callback function defintion
typedef int (__stdcall *tfnCallback)(USHORT, USHORT, USHORT, USHORT, bool);
TIPROHIDAPI_API int HIDRegisterTelephonyKeysCallback(USHORT hidIx, tfnCallback fnKeyCallback);


// prepared for future development 
// callback function for returning status of the thread: fnThreadStatus(USHORT nID, int nStatus, int nErrCode, int nRFUParam);
//
// Parameters:
//		nID;					// ID of the telephony device (supporting more then one telephony devices on the same PC)
//		nStatus;				// Status of the thread ( to be defined later... for instance 0x00 = Stopped, 0x01 = Running ... )
//		nErrCode;				// ErrCode for reported status ( to be defined later )
//		nRFUParam;				// Reserved for future use ( to be defined later )
// If Visual Studio 2010 or later is used callback can be defined inside of the class using std:bind
#if _MSC_VER > 1500
	typedef std::function<int(USHORT,int,int,int)> tfnThreadStatusEx;
	TIPROHIDAPI_API int HIDRegisterTelephonyKeysStatusCallbackEx(USHORT hidIx, tfnThreadStatusEx fnKeyStatusCallback);
#endif
typedef int (__stdcall *tfnThreadStatus)(USHORT, int, int, int);
TIPROHIDAPI_API int HIDRegisterTelephonyKeysStatusCallback(USHORT hidIx, tfnThreadStatus fnKeyStatusCallback);
#endif

/**
 * @brief Stops reporting keys of telephony device.
 * @param[in] hidIx Telephony device index
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDStopTelephonyKeyDevice(USHORT hidIx);

/**
 * @brief Returns telephony device path that can be used to access the device directly
 * @param[in] hidIx Telephony device index
 * @param[out] pString Pointer to user defined buffer (char *) where the path will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetTelephonyDevicePath(USHORT hidIx, PVOID pString, ULONG lBufferLen);

/****************************************************************************/
/*		Tiprobus / module functions and definitions                         */
/****************************************************************************/
//
// Module type definitions:
#define MODULE_TYPE_UNKNOWN				0x00 /**< Unknown module type - module not present */
#define MODULE_TYPE_MATRIX				0x01 /**< Keyboard module type */
#define MODULE_TYPE_SPEAKERBOX			0x02 /**< Speakerbox module type */
#define MODULE_TYPE_SPEAKERBOX_VOLUME	0x03 /**< Digital potentiometer module type */
#define MODULE_TYPE_CHAMELEON			0x04 /**< Chameleon module type */
#define MODULE_TYPE_HANDSET				0x05 /**< Handset module type */
#define MODULE_TYPE_BF15				0x06 /**< BeFREE 15 module type */
#define MODULE_TYPE_BF10				0x07 /**< BeFREE 10 module type */
#define MODULE_TYPE_BF10_AUDIO			0x08 /**< BeFREE audio 1 module type for BF10, BF20 and BF22 */
#define MODULE_TYPE_BF20				0x09 /**< BeFREE 20 module type */
#define MODULE_TYPE_BF20_AUDIO			0x0A /**< BeFREE audio 2 module type for BF10, BF20 and BF22 */
#define MODULE_TYPE_BF22				0x0B /**< BeFREE 22 module type */
#define MODULE_TYPE_BF22_LINEKEYS		0x0C /**< Linekeys module type for BF22*/
#define MODULE_TYPE_SPEAKERBOX_104		0x0D /**< Audio board module type */
#define MODULE_TYPE_BF07				0x0E /**< BeFREE 07 module type */
#define MODULE_TYPE_BF07_DIALPAD		0x0F /**< Dialpad module type for BF07 */
#define MODULE_TYPE_HANDSET_HAA			0x10 /**< Analogue switching handset (HAA) module type */
#define MODULE_TYPE_INTERCOM			0x11 /**< Intercom module type */
#define MODULE_TYPE_INTERCOM_EXT_SPK	0x12 /**< Intercom external speaker module type */
#define MODULE_TYPE_HANDSET_AM			0x13 /**< Module type for Handset (HUA/HUM) with audio monitoring */
#define MODULE_TYPE_HD_INTERNAL			0x14 /**< Internal Headset connection module type*/
#define MODULE_TYPE_UNSUPPORTED			0xFF /**< Unsuported module type - module not supported by API */

/**
 * @brief Returns version information of module in Tipro configuration
 * @param[in] nIndex Module index (zero based, from right to left)
 * @param[out] nVerMaj Major number
 * @param[out] nVerMin Minor number
 * @param[out] nVerBuild Build number
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetModuleVersion(int nIndex, unsigned char *nVerMaj, unsigned char *nVerMin, unsigned char *nVerBuild);

/**
 * @brief Returns version information of module in Tipro configuration
 * @param[in] nIndex Module index (zero based, from right to left)
 * @param[out] nVerMaj Major number
 * @param[out] nVerMin Minor number
 * @param[out] nVerBuild Build number
 * @param[out] nModuleType Module type
 * @param[out] nModuleID Module ID
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetModuleVersionEx(int nIndex, unsigned char *nVerMaj, unsigned char *nVerMin, unsigned char *nVerBuild, unsigned char *nModuleType, unsigned char *nModuleID);

/**
 * @brief Returns controller vesion information
 * @param[out] nVerMaj Major number
 * @param[out] nVerMin Minor number
 * @param[out] nVerBuild Build number
 * @param[out] nVerCustom Custom number
 * @param[out] nFWLevel Firmware level
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetControllerInfo(int *nVerMaj, int *nVerMin, int *nVerBuild,int *nVerCustom, int *nFWLevel);

/**
 * @brief Returns device manufacturer string
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] ulBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetManufacturerString(void *pString, unsigned long ulBufferLen);

/**
 * @brief Returns device product string
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] ulBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetProductString(void *pString, unsigned long ulBufferLen);

// USB Descriptor functions

/**
 * @brief Returns device indexed string descriptor
 * @param[in] lStrIx String index
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetStringDescriptor(ULONG lStrIx, PVOID pString, ULONG lBufferLen);

/**
 * @brief Returns indexed string descriptor of specific device
 * @param[in] hidIx Device index
 * @param[in] lStrIx String index
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] lBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetStringDescriptorEx(int hidIx, ULONG lStrIx, PVOID pString, ULONG lBufferLen );

// multiple controllers support

/**
 * @brief Returns number of detected devices
 * @return Number of detected devices
 */
TIPROHIDAPI_API int HIDGetNumOfDetectedDevices();

/**
 * @brief Select device to communicate with
 * @param[in] hidIx Device index
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDSelectDeviceEx(int hidIx);

/**
 * @brief Returns serial number of device
 * @param[out] pString Pointer to user defined buffer (char16_t *) where the string will be returned to
 * @param[in] ulBufferLen Length of a supplied buffer
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDGetSerialNumberString(void *pString, unsigned long ulBufferLen);

/**
 * @brief Sets keys to specified layer
 * @param[in] nLayer New active layer (KEY_LAYER1, KEY_LAYER2, KEY_LAYER3, KEY_LAYER4)
 * @return Command status: 0-OK, else error code
 */
TIPROHIDAPI_API int HIDSetKeyLayer(int nLayer);

#define KEY_LAYER1 0 /**< Layer 1 */
#define KEY_LAYER2 1 /**< Layer 2 */
#define KEY_LAYER3 2 /**< Layer 3 */
#define KEY_LAYER4 3 /**< Layer 4 */

#endif
