/**
 * @file Handset.h
 * 
 * Handset
 * 
 * This file includes functions for communication with Handset.
 * 
 * Handset
 * 
 * Tipro
 * 
 * support@tipro.si
 * 
 * Copyright 2021-2024
 */
 
#ifndef _TIPRO_TIPROHIDAPI_HANDSET_H_
#define _TIPRO_TIPROHIDAPI_HANDSET_H_

#include "TiproHidDevice.h"

#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************************/
/* STANDARD HANDSET DEFINES                                                 */
/****************************************************************************/

// microphone level (attenuation)
#define HXX_MIC_ATT_0					0x00 /**< Microphone attenuation level 0 (maximum) */
#define HXX_MIC_ATT_1					0x01 /**< Microphone attenuation level 1 */
#define HXX_MIC_ATT_2					0x02 /**< Microphone attenuation level 2 */
#define HXX_MIC_ATT_3					0x03 /**< Microphone attenuation level 3 */
#define HXX_MIC_ATT_4					0x04 /**< Microphone attenuation level 4 */
#define HXX_MIC_ATT_5					0x05 /**< Microphone attenuation level 5 */
#define HXX_MIC_ATT_6					0x06 /**< Microphone attenuation level 6 */
#define HXX_MIC_ATT_7					0x07 /**< Microphone attenuation level 7 */
#define HXX_MIC_ATT_8					0x08 /**< Microphone attenuation level 8 */
#define HXX_MIC_ATT_9					0x09 /**< Microphone attenuation level 9 */
#define HXX_MIC_ATT_10					0x0A /**< Microphone attenuation level 10 */
#define HXX_MIC_ATT_11					0x0B /**< Microphone attenuation level 11 */
#define HXX_MIC_ATT_12					0x0C /**< Microphone attenuation level 12 */
#define HXX_MIC_ATT_13					0x0D /**< Microphone attenuation level 13 */
#define HXX_MIC_ATT_14					0x0E /**< Microphone attenuation level 14 (minimum) */
#define	HXX_MIC_ATT_INIT_EEP			0xF0
#define HXX_MIC_MUTE					0xF3
#define HXX_MIC_RESTORE					0xF4


// paramters limited to usefull range
#define HXX_MIC_ATT_MIN					HXX_MIC_ATT_0
#define HXX_MIC_ATT_MAX					HXX_MIC_ATT_14

// microphone compression settings
#define HXX_MIC_COMPRESSION_OFF			0x10 /**< Microphone compression off */
#define HXX_MIC_COMPRESSION_ON			0x11 /**< Microphone compression on */

#define HXX_MIC_COMPRESSION_MIN				HXX_MIC_COMPRESSION_OFF
#define HXX_MIC_COMPRESSION_MAX				HXX_MIC_COMPRESSION_ON

// microphone noise threshold
#define HXX_MIC_THRESHOLD_HIGH			0x10 /**< Microphone noise threshold high */
#define HXX_MIC_THRESHOLD_MID			0x11 /**< Microphone noise threshold medium */
#define HXX_MIC_THRESHOLD_LOW			0x12 /**< Microphone noise threshold low */
#define HXX_MIC_THRESHOLD_LOWER			0x13 /**< Microphone noise threshold lower */
#define HXX_MIC_THRESHOLD_VERY_LOW		0x14 /**< Microphone noise threshold very low */
#define HXX_MIC_THRESHOLD_ILLEGAL		0xFF

#define HXX_MIC_THRESHOLD_MIN				HXX_MIC_THRESHOLD_HIGH
#define HXX_MIC_THRESHOLD_MAX				HXX_MIC_THRESHOLD_VERY_LOW

// Microphone optimal distance
#define HXX_MIC_DISTANCE_M5				0x07 /**< Microphone optimal distance -5 (minimum) */
#define HXX_MIC_DISTANCE_M4				0x08 /**< Microphone optimal distance -4 */
#define HXX_MIC_DISTANCE_M3				0x09 /**< Microphone optimal distance -3 */
#define HXX_MIC_DISTANCE_M2				0x0A /**< Microphone optimal distance -2 */
#define HXX_MIC_DISTANCE_M1				0x0B /**< Microphone optimal distance -1 */
#define HXX_MIC_DISTANCE_0 				0x0C /**< Microphone optimal distance 0 */
#define HXX_MIC_DISTANCE_P1				0x0D /**< Microphone optimal distance +1 */
#define HXX_MIC_DISTANCE_P2				0x0E /**< Microphone optimal distance +2 */
#define HXX_MIC_DISTANCE_P3				0x0F /**< Microphone optimal distance +3 */
#define HXX_MIC_DISTANCE_P4				0x10 /**< Microphone optimal distance +4 */
#define HXX_MIC_DISTANCE_P5				0x11 /**< Microphone optimal distance +5 (maximum)*/

#define HXX_MIC_DISTANCE_MIN			HXX_MIC_DISTANCE_M5
#define HXX_MIC_DISTANCE_MAX			HXX_MIC_DISTANCE_P5

// Microphone mode
#define HXX_MICAMP_MODE_LOG				0x00 /**< Microphone in logaritmic mode */
#define HXX_MICAMP_MODE_LIN				0x04 /**< Microphone in linear mode */

// speaker
#define HXX_SPEAKER_LEVEL_P2			0x00 /**< Speaker level +2 (maximum) */
#define HXX_SPEAKER_LEVEL_P1			0x01 /**< Speaker level +1 */
#define HXX_SPEAKER_LEVEL_0				0x02 /**< Speaker level 0 */
#define HXX_SPEAKER_LEVEL_M1			0x03 /**< Speaker level -1 */
#define HXX_SPEAKER_LEVEL_M2			0x04 /**< Speaker level -2 */
#define HXX_SPEAKER_LEVEL_M3			0x05 /**< Speaker level -3 */
#define HXX_SPEAKER_LEVEL_M4			0x06 /**< Speaker level -4 */
#define HXX_SPEAKER_LEVEL_M5			0x07 /**< Speaker level -5 */
#define HXX_SPEAKER_LEVEL_M6			0x08 /**< Speaker level -6 */
#define HXX_SPEAKER_LEVEL_M7			0x09 /**< Speaker level -7 */
#define HXX_SPEAKER_LEVEL_M8			0x0A /**< Speaker level -8 (minimum)*/
#define HXX_SPEAKER_LEVEL_ILLEGAL		0xFF
#define HXX_SPEAKER_MUTE				0xF3
#define HXX_SPEAKER_RESTORE				0xF4

#define HXX_SPEAKER_LEVEL_MIN	HXX_SPEAKER_LEVEL_P2
#define HXX_SPEAKER_LEVEL_MAX	HXX_SPEAKER_LEVEL_M8


/****************************************************************************/
/* HAA HANDSET DEFINES														*/
/****************************************************************************/
// Microphone optimal distance
#define HAA_MIC_DISTANCE_M5				0x07 /**< Microphone optimal distance -5 (minimum) */
#define HAA_MIC_DISTANCE_M4				0x08 /**< Microphone optimal distance -4 */
#define HAA_MIC_DISTANCE_M3				0x09 /**< Microphone optimal distance -3 */
#define HAA_MIC_DISTANCE_M2				0x0A /**< Microphone optimal distance -2 */
#define HAA_MIC_DISTANCE_M1				0x0B /**< Microphone optimal distance -1 */
#define HAA_MIC_DISTANCE_0				0x0C /**< Microphone optimal distance 0 */
#define HAA_MIC_DISTANCE_P1				0x0D /**< Microphone optimal distance +1 */
#define HAA_MIC_DISTANCE_P2				0x0E /**< Microphone optimal distance +2 */
#define HAA_MIC_DISTANCE_P3				0x0F /**< Microphone optimal distance +3 */
#define HAA_MIC_DISTANCE_P4				0x10 /**< Microphone optimal distance +4 */
#define HAA_MIC_DISTANCE_P5				0x11 /**< Microphone optimal distance +5 (maximum)*/

#define HAA_MIC_DISTANCE_MIN			HAA_MIC_DISTANCE_M5
#define HAA_MIC_DISTANCE_MAX			HAA_MIC_DISTANCE_P5

// Handset sidetone level
#define HAA_SIDETONE_LEVEL_M1					0x00 /**< Handset sidetone level -1 (maximum) */
#define HAA_SIDETONE_LEVEL_M2					0x01 /**< Handset sidetone level -2 */
#define HAA_SIDETONE_LEVEL_M3					0x02 /**< Handset sidetone level -3 */
#define HAA_SIDETONE_LEVEL_M4					0x03 /**< Handset sidetone level -4 */
#define HAA_SIDETONE_LEVEL_M5					0x04 /**< Handset sidetone level -5 */
#define HAA_SIDETONE_LEVEL_M6					0x05 /**< Handset sidetone level -6 */
#define HAA_SIDETONE_LEVEL_M7					0x06 /**< Handset sidetone level -7 */
#define HAA_SIDETONE_LEVEL_M8					0x07 /**< Handset sidetone level -8 */
#define HAA_SIDETONE_LEVEL_M9					0x08 /**< Handset sidetone level -9 (minimum) */
#define HAA_SIDETONE_OFF						0xFE
#define HAA_SIDETONE_ILLEGAL					0xFF
// limits to check
#define HAA_SIDETONE_MIN						HAA_SIDETONE_LEVEL_M1
#define HAA_SIDETONE_MAX						HAA_SIDETONE_LEVEL_M9

/****************************************************************************/
/*		Handset with audio monitoring defines								*/
/****************************************************************************/

// paramters for near-end microphone signal attenuation (HXX_EXTCMD_AM_HS_MIC_MIX)
#define HXX_AM_HS_MIC_MIX_ATT_0			0x00 /**< Near-end microphone signal attenuation level 0 (maximum) */
#define HXX_AM_HS_MIC_MIX_ATT_1			0x01 /**< Near-end microphone signal attenuation level 1 */
#define HXX_AM_HS_MIC_MIX_ATT_2			0x02 /**< Near-end microphone signal attenuation level 2 */
#define HXX_AM_HS_MIC_MIX_ATT_3			0x03 /**< Near-end microphone signal attenuation level 3 */
#define HXX_AM_HS_MIC_MIX_ATT_4			0x04 /**< Near-end microphone signal attenuation level 4 */
#define HXX_AM_HS_MIC_MIX_ATT_5			0x05 /**< Near-end microphone signal attenuation level 5 */
#define HXX_AM_HS_MIC_MIX_ATT_6			0x06 /**< Near-end microphone signal attenuation level 6 */
#define HXX_AM_HS_MIC_MIX_ATT_7			0x07 /**< Near-end microphone signal attenuation level 7 */
#define HXX_AM_HS_MIC_MIX_ATT_8			0x08 /**< Near-end microphone signal attenuation level 8 */
#define HXX_AM_HS_MIC_MIX_ATT_9			0x09 /**< Near-end microphone signal attenuation level 9 */
#define HXX_AM_HS_MIC_MIX_ATT_10		0x0A /**< Near-end microphone signal attenuation level 10 */
#define HXX_AM_HS_MIC_MIX_ATT_11		0x0B /**< Near-end microphone signal attenuation level 11 */
#define HXX_AM_HS_MIC_MIX_ATT_12		0x0C /**< Near-end microphone signal attenuation level 12 */
#define HXX_AM_HS_MIC_MIX_ATT_13		0x0D /**< Near-end microphone signal attenuation level 13 */
#define HXX_AM_HS_MIC_MIX_ATT_14		0x0E /**< Near-end microphone signal attenuation level 14 (maximum) */
#define	HXX_AM_HS_MIC_MIX_INIT_EEP		0xF0
#define HXX_AM_HS_MIC_MIX_GAIN_INC		0xF1
#define HXX_AM_HS_MIC_MIX_GAIN_DEC		0xF2
#define HXX_AM_HS_MIC_MIX_MUTE			0xF3
#define HXX_AM_HS_MIC_MIX_RESTORE		0xF4

#define HXX_AM_HS_MIC_MIX_ATT_MIN		HXX_AM_HS_MIC_MIX_ATT_0
#define HXX_AM_HS_MIC_MIX_ATT_MAX		HXX_AM_HS_MIC_MIX_ATT_14

// parameters for audio monitoring output signal amplification (HXX_EXTCMD_AM_SPK_AMP)
#define HXX_AM_SPK_LEVEL_P4			0x00 /**< Audio monitoring output signal amplification level +4 (maximum) */
#define HXX_AM_SPK_LEVEL_P3			0x01 /**< Audio monitoring output signal amplification level +3 */
#define HXX_AM_SPK_LEVEL_P2			0x02 /**< Audio monitoring output signal amplification level +2 */
#define HXX_AM_SPK_LEVEL_P1			0x03 /**< Audio monitoring output signal amplification level +1 */
#define HXX_AM_SPK_LEVEL_0			0x04 /**< Audio monitoring output signal amplification level 0 */
#define HXX_AM_SPK_LEVEL_M1			0x05 /**< Audio monitoring output signal amplification level -1 */
#define HXX_AM_SPK_LEVEL_M2			0x06 /**< Audio monitoring output signal amplification level -2 */
#define HXX_AM_SPK_LEVEL_M3			0x07 /**< Audio monitoring output signal amplification level -3 */
#define HXX_AM_SPK_LEVEL_M4			0x08 /**< Audio monitoring output signal amplification level -4 */
#define HXX_AM_SPK_LEVEL_M5			0x09 /**< Audio monitoring output signal amplification level -5 */
#define HXX_AM_SPK_LEVEL_M6			0x0A /**< Audio monitoring output signal amplification level -6 (minimum) */
#define	HXX_AM_SPK_LEVEL_INIT_EEP	0xF0
#define HXX_AM_SPK_LEVEL_INC		0xF1
#define HXX_AM_SPK_LEVEL_DEC		0xF2
#define HXX_AM_SPK_MUTE				0xF3
#define HXX_AM_SPK_RESTORE			0xF4

#define HXX_AM_SPK_LEVEL_MIN		HXX_AM_SPK_LEVEL_P4
#define HXX_AM_SPK_LEVEL_MAX		HXX_AM_SPK_LEVEL_M6


/****************************************************************************/
/*		Handset functions													*/
/****************************************************************************/

/**
 * @brief Set compression for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nCompression Microphone compression
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicrophoneCompression (TiproHidDevice *device, int nHandsetNum, int nCompression);

/**
 * @brief Sets threshold level for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nThreshold Microphone threshold
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicrophoneThreshold (TiproHidDevice *device, int nHandsetNum, int nThreshold);

/**
 * @brief Sets level of the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Microphone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicrophoneLevel (TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Sets level of the microphone signal, optionaly store it to EEPROM
 * @param device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Microphone level
 * @param[in] bSetInit Write initial state to EEPROM
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicrophoneLevelEx(TiproHidDevice *device, int nHandsetNum, int nLevel, bool bSetInit);

/**
 * @brief Returns compression for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nCompression Microphone compression
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetMicrophoneCompression (TiproHidDevice *device, int nHandsetNum, int *nCompression);

/**
 * @brief Returns threshold for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nThreshold Microphone threshold
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetMicrophoneThreshold (TiproHidDevice *device, int nHandsetNum, int *nThreshold);

/**
 * @brief Returns compression for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nLevel Microphone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetMicrophoneLevel (TiproHidDevice *device, int nHandsetNum, int *nLevel);

/**
 * @brief Sets the level of the handset speaker
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Speaker level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetSpeakerLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Returns the level of the handset speaker
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nLevel Speaker level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetSpeakerLevel(TiproHidDevice *device, int nHandsetNum, int *nLevel);

/**
 * @brief Resets the HID telephony events and sends hook and ptt button events
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nTelephonyEvents Telephony events to be reset (1-hook, 2-mute)
 * @param[in] nTelephonyKeys Should telephony keys be reset (0-false; else true)
 * @param[in] nTelephonyButtons Should telephony program keys be reset (0-false, else true)
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSendHookPttEvents(TiproHidDevice *device, int nHandsetNum, int nTelephonyEvents, int nTelephonyKeys, int nTelephonyButtons);

/**
 * @brief Sets the microphone optimal distance
 * @param[in] device Device
 * @param[in] nHandsetNum Address of the module
 * @param[in] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int nDistance);

/**
 * @brief Set Microphone amplifier mode (Linear/Logarithmic)
 * @param[in] device Device
 * @param[in] nHandsetNum Address of the module
 * @param[in] nMicAmpMode Microphone amplifier mode (LIN/LOG)
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicAmpMode(TiproHidDevice *device, int nHandsetNum, int nMicAmpMode);

/**
 * @brief Returns the optimal distance of the microphone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int *nDistance);

/**
 * @brief Sets and stores the optimal distance for the microphone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetStoreMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int nDistance);

/**
 * @brief Sets and stores compression for the microphone signal in EEPROM
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nCompression Microphone compression
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetStoreMicrophoneCompression(TiproHidDevice *device, int nHandsetNum, int nCompression);

/**
 * @brief Sets and stores thresholed for the microphone signal
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nThreshold Microphone threshold
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetStoreMicrophoneThreshold(TiproHidDevice *device, int nHandsetNum, int nThreshold);

/**
 * @brief Sets and stores microphone signal level
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Microphone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetStoreMicrophoneLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Sets and stores the level of the handset speaker
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Speaker level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetStoreSpeakerLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);


/****************************************************************************/
/*		Handset functions ( HAA specific )									*/
/****************************************************************************/

/**
 * @brief Sets and stores the optimal distance for the microphone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAStoreMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int nDistance);

/**
 * @brief Sets and stores the level of the handset sidetone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Handset sidetone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAStoreSidetoneLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Enables setting microphone and speaker level when handset is on-hook
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAEnableOnHookSettings(TiproHidDevice *device, int nHandsetNum);

/**
 * @brief Sets the optimal distance for the microphone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code 
 */
int TiproHandsetHAASetMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int nDistance);

/**
 * @brief Returns the optimal distance of the microphone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nDistance Microphone optimal distance
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAGetMicrophoneOptimalDistance(TiproHidDevice *device, int nHandsetNum, int *nDistance);

/**
 * @brief Sets the level of the handset sidetone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Handset sidetone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAASetSidetoneLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Turn sidetone On or Off
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] bOn Handset sidetone On/Off
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAASetHandsetSidetone(TiproHidDevice *device, int nHandsetNum, bool bOn);

/**
 * @brief Returns the current level of the handset sidetone
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nLevel Handset sidetone level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAGetSidetoneLevel(TiproHidDevice *device, int nHandsetNum, int *nLevel);

/**
 * @brief Forces HAA state to HFREE regardless of the hook state
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAASetForceHFREE(TiproHidDevice *device, int nHandsetNum);

/**
 * @brief Clears forced HAA state
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAClearForceHFREE(TiproHidDevice *device, int nHandsetNum);

/**
 * @brief Emulates On-Hook action. The handset has to be programmed properly
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAEmulateOnHook(TiproHidDevice *device, int nHandsetNum);

/**
 * @brief Emulates Off-Hook action. The handset has to be programmed properly
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetHAAEmulateOffHook(TiproHidDevice *device, int nHandsetNum);


/****************************************************************************/
/*		Handset functions ( Audio monitoring version specific )				*/
/****************************************************************************/

/**
 * @brief Sets audio monitoring amplification level
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Audio monioring amplification level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetAMLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Returns audio monitoring amplification level
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nLevel Audio monioring amplification level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetAMLevel(TiproHidDevice *device, int nHandsetNum, int *nLevel);

/**
 * @brief Sets near-end microphone signal attenuation level
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[in] nLevel Near-end microphone signal attenuation level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetSetMicMixLevel(TiproHidDevice *device, int nHandsetNum, int nLevel);

/**
 * @brief Returns near-end microphone signal attenuation level
 * @param[in] device Device
 * @param[in] nHandsetNum Selected handset module number
 * @param[out] nLevel Near-end microphone signal attenuation level
 * @return Command status: 0-OK, else error code
 */
int TiproHandsetGetMicMixLevel(TiproHidDevice *device, int nHandsetNum, int *nLevel);



#ifdef __cplusplus
}
#endif

#endif
